<?php

namespace SV\SignupAbuseBlocking\XF\Admin\Controller;

use SV\SignupAbuseBlocking\Entity\UserRegistrationLog;
use XF\Entity\User as UserEntity;
use XF\Mvc\Entity\AbstractCollection;
use XF\Mvc\Entity\Entity;
use XF\Mvc\ParameterBag;
use XF\Mvc\Reply\AbstractReply;
use XF\Mvc\Reply\Exception;
use SV\SignupAbuseBlocking\Finder\UserRegistrationLog as UserRegistrationLogFinder;
use SV\SignupAbuseBlocking\Repository\UserRegistrationLog as UserRegistrationLogRepo;
use XF\Repository\User as UserRepo;
use XF\Util\Arr as ArrUtil;
use function strlen;

/**
 * Class Log
 *
 * @package SV\SignupAbuseBlocking\XF\Admin\Controller
 */
class Log extends XFCP_Log
{
    protected function getUserRegistrationFilterInputForSv(string &$usernames = null) : array
    {
        $filters = [];
        $usernames = '';

        $em = $this->em();
        $input = $this->filter([
            'details' => 'str',

            'user_ids' => 'array-uint',
            'usernames' => 'str',

            'log_date_start' => '?datetime',
            'log_date_end' => '?datetime'
        ]);

        if (strlen($input['details']) !== 0)
        {
            $filters['details'] = $input['details'];
        }

        if ($input['log_date_start'])
        {
            $filters['log_date_start'] = $input['log_date_start'];
        }

        if ($input['log_date_end'])
        {
            $filters['log_date_end'] = $input['log_date_end'];
        }

        $users = null;
        if ($input['user_ids'])
        {
            $users = $em->find('XF:User', $input['user_ids']);
        }
        else if ($input['usernames'])
        {
            /** @var UserRepo $userRepo */
            $userRepo = $this->repository('XF:User');
            $users = $userRepo->getUsersByNames(ArrUtil::stringToArray($input['usernames'], '/,\s*/'));
        }

        if ($users instanceof UserEntity)
        {
            $users = $em->getBasicCollection([
                $users->getEntityId() => $users
            ]);
        }

        if ($users instanceof AbstractCollection && $users->count())
        {
            $filters['user_ids'] = [];
            $usernamesArr = [];

            /** @var UserEntity $user */
            foreach ($users AS $user)
            {
                $filters['user_ids'][] = $user->user_id;
                $usernamesArr[] = $user->username;
            }

            $usernames = \implode(',', $usernamesArr);
        }

        return $filters;
    }

    protected function applyUserRegistrationFilterForSv(UserRegistrationLogFinder $logFinder, array $filters)
    {
        if (!\count($filters))
        {
            return;
        }

        if (!empty($filters['user_ids']))
        {
            $logFinder->where('user_id', $filters['user_ids']);
        }

        if (\array_key_exists('details', $filters) && strlen($filters['details']) !== 0)
        {
            $logFinder->where(
                'details',
                'LIKE',
                $logFinder->escapeLike($filters['details'], '%?%')
            );
        }

        if (!empty($filters['log_date_start']))
        {
            $logFinder->where('log_date', '>=', $filters['log_date_start']);
        }

        if (!empty($filters['log_date_end']))
        {
            $logFinder->where('log_date', '<=', $filters['log_date_end']);
        }
    }

    public function actionUserRegistration(ParameterBag $parameterBag): AbstractReply
    {
        /** @noinspection PhpUndefinedFieldInspection */
        if ($parameterBag->user_registration_log_id)
        {
            return $this->rerouteController(__CLASS__, 'userRegistrationView', $parameterBag);
        }

        $linkParams = $this->getUserRegistrationFilterInputForSv($usernames);
        if ($this->isPost() && $this->filter('apply', 'bool'))
        {
            return $this->redirect($this->buildLink('logs/user-registration', null, $linkParams));
        }

        $page = $this->filterPage();
        $perPage = 20;

        $logFinder = $this->getUserRegistrationLogRepo()
            ->findUserRegistrationLogsForList()
            ->limitByPage($page, $perPage);
        $this->applyUserRegistrationFilterForSv($logFinder, $linkParams);

        $logs = $logFinder->fetch();
        /** @var UserRegistrationLog $log */
        foreach ($logs as $log)
        {
            $log->setOption('linkUsernameInDetails', false);
        }
        $viewParams = [
            'entries' => $logs,

            'usernames'   => $usernames,

            'pageLink'   => 'logs/user-registration',

            'page'       => $page,
            'perPage'    => $perPage,
            'total'      => $logFinder->total(),

            'linkParams' => $linkParams,
        ];

        return $this->view('SV\SignupAbuseBlocking\XF:Log\UserRegistration\Listing', 'svSignupAbuseBlocking_log_user_registration_list', $viewParams);
    }

    public function actionUserRegistrationView(ParameterBag $parameterBag): AbstractReply
    {
        /** @noinspection PhpUndefinedFieldInspection */
        $entry = $this->assertUserRegistrationLogExists($parameterBag->user_registration_log_id);

        // stop xdebug overloading the dump...
        @ini_set("xdebug.overload_var_dump", "off");

        $viewParams = [
            'entry' => $entry
        ];
        return $this->view(
            'SV\SignupAbuseBlocking\XF:Log\UserRegistration\View',
            'svSignupAbuseBlocking_log_user_registration_view',
            $viewParams
        );
    }

    /** @noinspection PhpMissingReturnTypeInspection */
    public function actionSpamTriggerView(ParameterBag $params)
    {
        // stop xdebug overloading the dump...
        @ini_set("xdebug.overload_var_dump", "off");
        return parent::actionSpamTriggerView($params);
    }

    /**
     * @param int   $id
     * @param array $with
     * @param ?string  $phraseKey
     * @return Entity|UserRegistrationLog
     * @throws Exception
     */
    protected function assertUserRegistrationLogExists(int $id, array $with = [], string $phraseKey = 'requested_log_entry_not_found')
    {
        return $this->assertRecordExists('SV\SignupAbuseBlocking:UserRegistrationLog', $id, $with, $phraseKey);
    }

    protected function getUserRegistrationLogRepo() : UserRegistrationLogRepo
    {
        $repo = $this->repository('SV\SignupAbuseBlocking:UserRegistrationLog');
        assert($repo instanceof UserRegistrationLogRepo);
        return $repo;
    }
}