<?php

namespace SV\SignupAbuseBlocking\Spam\Checker\User;

use SV\StandardLib\BypassAccessStatus;
use XF\Entity\User as UserEntity;
use XF\Spam\Checker\AbstractProvider;
use XF\Spam\Checker\UserCheckerInterface;
use XF\Spam\ContentChecker as ContentSpamChecker;

/**
 * Class UserField
 *
 * @package SV\SignupAbuseBlocking\Spam\Checker
 */
class UserField extends AbstractProvider implements UserCheckerInterface
{
    protected function getType(): string
    {
        return 'SignupAbuseUserField';
    }

    /**
     * @param UserEntity $user
     * @param array $extraParams
     * 
     * @throws \Exception
     */
    public function check(UserEntity $user, array $extraParams = [])
    {
        $profile = $user->Profile;
        if (!$profile)
        {
            $this->logDecision('allowed');
            return;
        }

        $message = '';

        $customFields = $profile->custom_fields;
        $spammableFieldTypes = $this->getSpammableFieldTypes();

        foreach ($customFields->getFieldValues() AS $fieldId => $value)
        {
            $definition = $customFields->getDefinition($fieldId);
            if (!\in_array($definition['field_type'], $spammableFieldTypes))
            {
                continue;
            }

            if (!\is_string($value) || trim($value) === '')
            {
                continue;
            }

            $message .= ' ' . $value;
        }

        foreach ($profile->structure()->columns AS $column => $columnData)
        {
            if (!\array_key_exists('checkForSpamPhrases', $columnData) || $columnData['checkForSpamPhrases'] !== true)
            {
                continue;
            }

            $value = $profile->{$column};
            if (!\is_string($value))
            {
                continue;
            }

            $message .= ' ' . $value;
        }

        $message = \trim($message);
        if (\strlen($message) === 0)
        {
            $this->logDecision('allowed');
            return;
        }

        $contentSpamChecker = $this->contentSpamChecker();
        $contentSpamChecker->check($user, $message, [
            'content_type' => 'user' // see \XF\Spam\Checker\Akismet for content types
        ]);

        $accesser = new BypassAccessStatus();
        $decisions = $accesser->getPrivate($contentSpamChecker, 'decisions')();
        $details = $accesser->getPrivate($contentSpamChecker, 'details')();
        $params = $accesser->getPrivate($contentSpamChecker, 'params')();

        foreach ($decisions AS $type => $decision)
        {
            $this->checker->logDecision($type, $decision);
        }
        foreach ($details AS $type => $detail)
        {
            $phrase = $detail['phrase'];
            $data = [];
            if (\array_key_exists('data', $detail))
            {
                $data = $detail['data'];
            }

            $this->checker->logDetail($type, $phrase, $data);
        }
        foreach ($params AS $key => $param)
        {
            $this->checker->logParam($key, $param);
        }
    }

    /**
     * @return string[]
     */
    protected function getSpammableFieldTypes(): array
    {
        return ['textbox', 'textarea', 'bbcode'];
    }

    /**
     * @param UserEntity $user
     * @param array      $extraParams
     */
    public function submit(UserEntity $user, array $extraParams = [])
    {
    }

    protected function contentSpamChecker(): ContentSpamChecker
    {
        return $this->app()->spam()->contentChecker();
    }
}