<?php
/**
 * @noinspection PhpMissingParamTypeInspection
 * @noinspection PhpMissingReturnTypeInspection
 */

namespace SV\SignupAbuseBlocking\Repository;

use SV\SignupAbuseBlocking\Entity\Log;
use SV\SignupAbuseBlocking\Globals;
use SV\SignupAbuseBlocking\MultipleAccount\DetectionMethodEmailLink;
use SV\SignupAbuseBlocking\MultipleAccount\DetectionMethodLegacy;
use SV\SignupAbuseBlocking\MultipleAccount\DetectionRecord;
use SV\SignupAbuseBlocking\MultipleAccount\DetectionMethodCookie;
use SV\SignupAbuseBlocking\MultipleAccount\DetectionMethodIp;
use SV\SignupAbuseBlocking\Entity\LogEvent;
use SV\SignupAbuseBlocking\Entity\ReportData;
use SV\SignupAbuseBlocking\Entity\Token as TokenEntity;
use SV\SignupAbuseBlocking\XF\Entity\Report;
use SV\SignupAbuseBlocking\XF\Entity\User;
use XF\Entity\ReportComment;
use XF\Mvc\Entity\ArrayCollection;
use XF\Mvc\Entity\Repository;
use XF\Phrase;
use XF\PrintableException;
use XF\Service\Report\Creator as ReportCreator;
use SV\SignupAbuseBlocking\Util\Ip as Ip;

class MultipleAccount extends Repository
{
    const MATCHING_MODE_OR          = 0;
    const MATCHING_MODE_AND         = 1;
    const MATCHING_MODE_COOKIE_ONLY = 2;

    /**
     * @param DetectionRecord[] $dataCollection
     * @throws PrintableException
     * @throws \Throwable
     * @throws \XF\Db\Exception
     */
    public function postRegistrationMultipleAccountDetection(array $dataCollection)
    {
        if (empty($dataCollection))
        {
            return;
        }

        if ($this->options()->svSockSignupCheckOnRegister ?? false)
        {
            $this->processMultipleAccountDetection(\XF::visitor(), $dataCollection, 'register');
        }
    }

    /**
     * @param \XF\Entity\User   $currentUser
     * @param DetectionRecord[] $dataCollection
     * @param string            $detectionAction
     * @throws PrintableException
     * @throws \Throwable
     * @throws \XF\Db\Exception
     */
    public function processMultipleAccountDetection(\XF\Entity\User $currentUser, array $dataCollection, $detectionAction)
    {
        $reportCreator = $this->getUserForReportCreation();

        if ($reportCreator !== null && $dataCollection)
        {
            \XF::asVisitor($reportCreator, function () use ($currentUser, $dataCollection, $detectionAction) {
                $this->processMultipleAccountDetectionInternal($currentUser, $dataCollection, $detectionAction);
            });
        }
    }

    /**
     * @return null|User
     */
    public function getUserForReportCreation()
    {
        $userId = (int)($this->options()->svSockSignupCheckReportingUser ?? 0);
        if (!$userId)
        {
            return null;
        }

        /** @var User $reportCreator */
        $reportCreator = $this->em->find('XF:User', $userId);
        if (!$reportCreator)
        {
            $e = new \LogicException("Multiple Account report; UserId set to an invalid user (ID: $userId). Aborting reporting multiple accounts.");
            if (\XF::$developmentMode)
            {
                throw $e;
            }
            \XF::logException($e);
        }

        return $reportCreator;
    }

    /**
     * @param \XF\Entity\User   $currentUser
     * @param DetectionRecord[] $dataCollection
     * @param string            $detectionAction
     * @throws PrintableException
     * @throws \Throwable
     * @throws \XF\Db\Exception
     */
    protected function processMultipleAccountDetectionInternal(\XF\Entity\User $currentUser, array $dataCollection, $detectionAction)
    {
        /** @var User $currentUser */
        if (!$dataCollection)
        {
            return;
        }
        $userId = $currentUser->user_id;
        if (!$userId)
        {
            // wat
            return;
        }

        // record the detection events
        $dataCollectionToReport = [];

        /** @var LogEvent $logEvent */
        $logEvent = $this->app()->em()->create('SV\SignupAbuseBlocking:LogEvent');
        $logEvent->detection_action = $detectionAction;
        $logEvent->triggering_user_id = $userId;
        $logEvent->username = $currentUser->username;
        $logEvent->hydrateRelation('User', $currentUser);
        $logEvent->report_data_id = null;

        foreach ($dataCollection AS $data)
        {
            $user = $data->user;

            $data->log = $logEvent->createLog($data->methods, $user, $data->token);

            if ($user->Profile->multiple_account_detection_alertable)
            {
                $dataCollectionToReport[$user->user_id] = $data;
            }
        }

        $db = $this->db();
        $userIdsQuoted = $db->quote(\array_unique([$userId] + array_keys($dataCollectionToReport)));
        if (\strlen($userIdsQuoted) === 0)
        {
            // wat
            return;
        }

        $db->beginTransaction();

        // need some locking to avoid wonky race conditions
        $db->query("select user_id from xf_user where user_id in ({$userIdsQuoted}) order by user_id for update");

        $logEvent->hydrateRelation('Logs', new ArrayCollection($dataCollectionToReport));
        $logEvent->save(true, false);
        $logEvent->clearCascadeSaveHack();

        // if we don't have at least a pair of users, there is nothing to REPORT
        // but we record matching records anyway
        if (!$dataCollectionToReport || !$currentUser->Profile->multiple_account_detection_alertable)
        {
            $db->commit();

            return;
        }

        // Find the first reported event for the reportable user id set
        // don't need the token, as detectMultipleAccount should have loaded it IF it was alertable
        $reportDataId = null;

        $matchingRow = $db->fetchRow("
            select distinct `event`.report_data_id
            from xf_sv_multiple_account_log as `log`
            join xf_sv_multiple_account_event as `event` on `event`.event_id = `log`.event_id
            join xf_sv_multiple_account_report_data as `reportData` on (`reportData`.report_data_id = `event`.report_data_id and `reportData`.`active` = 1)
            join xf_user_profile as `userProfile` on `userProfile`.user_id = `log`.user_id
            where (`log`.user_id in ({$userIdsQuoted}) or `event`.triggering_user_id in ({$userIdsQuoted}))
                  AND `log`.active = 1
                  AND `reportData`.active = 1
            order by `event`.detection_date ASC, `event`.triggering_user_id ASC, `log`.user_id ASC
            limit 1
        ");

        // burn the log event <-> report data link

        if ($matchingRow)
        {
            $reportDataId = $matchingRow['report_data_id'];

            if ($reportDataId && $this->hasIgnoredUserSetForReportEvent($logEvent, $reportDataId))
            {
                $db->commit();

                return;
            }
        }

        if (!$reportDataId)
        {
            /** @var ReportData $reportData */
            $reportData = $this->em->create('SV\SignupAbuseBlocking:ReportData');
            $reportData->save(true, false);
            $reportDataId = $reportData->report_data_id;
        }
        $logEvent->report_data_id = $reportDataId;
        $reportData = $logEvent->ReportData;
        $reportData->log_count += $logEvent->log_count;
        $logEvent->saveIfChanged($saved, true, false);
        $reportData->saveIfChanged($saved, true, false);

        $db->commit();

        // determine if any of the users have been marked as not-alertable
        $userReportingTotals = $db->fetchAllKeyed("
            select user_id, count(log_id) as `activeLogs`
            from xf_sv_multiple_account_log as `log`
            join xf_sv_multiple_account_event as `event` on `event`.event_id = `log`.event_id
            where (`log`.user_id in ({$userIdsQuoted}) or `event`.triggering_user_id in ({$userIdsQuoted}))
                  AND `log`.active = 1
                  AND `log`.is_alertable = 1
                  AND `event`.report_data_id = ? 
                  AND `log`.event_id <> ?
            group by user_id
            having `activeLogs` > 0
        ", 'user_id', [$logEvent->report_data_id, $logEvent->event_id]);

        foreach ($dataCollectionToReport as $userId => $data)
        {
            if (isset($userReportingTotals[$userId]) && $userReportingTotals[$userId]['activeLogs'] == 0)
            {
                unset($dataCollectionToReport[$userId]);
            }
        }

        // no log entries to report
        if (!$dataCollectionToReport)
        {
            return;
        }

        $logEvent->User->setOption('svMultiAccountLogEvent', $logEvent);
        $createdReporting = $this->createNewReportEntries($logEvent);
        try
        {
            $reportData->saveIfChanged($saved);
        }
        /** @noinspection PhpRedundantCatchClauseInspection */
        catch (\XF\Db\DuplicateKeyException $e)
        {
            // race condition; something else is active. make sure we save this record.
            $reportData->set('active', null, ['forceSet' => true]);
            $reportData->save();
            // do not bump
            return;
        }

        // determine if this event has recently been seen before to avoid superfluous duplication
        $bumpDedupeFilter = \XF::options()->svSockDedupeFilter ?? [];
        if ($logEvent->log_count === 1 && !empty($bumpDedupeFilter['seenFilter']))
        {
            $cutOff = (int)($bumpDedupeFilter['cutOff'] ?? 0);
            if ($cutOff)
            {
                $cutOff = \XF::$time - $cutOff * 86400;
            }
            if ($this->seenPair($logEvent, $cutOff))
            {
                return;
            }
        }

        $this->bumpReportEntries($createdReporting, $logEvent);
        $reportData->saveIfChanged($saved);
    }

    /**
     * @param LogEvent $logEvent
     * @param int      $cutoff
     * @return bool
     */
    protected function seenPair(LogEvent $logEvent, $cutoff)
    {
        /** @var Log $firstLog */
        $firstLog = $logEvent->Logs->first();

        return (bool)$this->db()->fetchOne("
            select distinct `event`.report_data_id
            from xf_sv_multiple_account_log as `log`
            join xf_sv_multiple_account_event as `event` on `event`.event_id = `log`.event_id
            join xf_sv_multiple_account_report_data as `reportData` on (`reportData`.report_data_id = `event`.report_data_id and `reportData`.`active` = 1)
            join xf_user_profile as `userProfile` on `userProfile`.user_id = `log`.user_id
            where 
                  `log`.active = 1 AND 
                  `event`.event_id <> ? AND 
                  `event`.detection_date > ? AND
                  ((`event`.triggering_user_id = ? and `log`.user_id = ?) or (`event`.triggering_user_id = ? and `log`.user_id = ?))
            order by `event`.detection_date ASC, `event`.triggering_user_id ASC, `log`.user_id ASC
            limit 1
        ", [$logEvent->event_id, $cutoff, $logEvent->triggering_user_id, $firstLog->user_id, $firstLog->user_id, $logEvent->triggering_user_id]);
    }

    /**
     * @param LogEvent $logEvent
     * @return array
     * @throws \Throwable
     * @throws \XF\Db\Exception
     */
    protected function createNewReportEntries(LogEvent $logEvent)
    {
        $reportData = $logEvent->ReportData;
        $options = \XF::options();
        $created = [];
        if (!$reportData->Report && ($options->svSockSignupCheckReport ?? false))
        {
            /** @var Report $report */
            $report = $this->reportCreator($logEvent);
            if ($report)
            {
                if ($report instanceof \XF\Entity\Report)
                {
                    if (!$reportData->exists())
                    {
                        // report may be a new report, or an existing one. Rebind as required
                        $this->db()->query('update xf_sv_multiple_account_report_data set active = 0 where report_id = ? and active = 1', $report->report_id);
                    }
                    $reportData->report_id = $report->report_id;
                    $reportData->hydrateRelation('Report', $report);
                }
                $created['report'] = $report;
            }
        }

        return $created;
    }

    /**
     * @param LogEvent $logEvent
     * @return null|Report|\XF\Entity\Thread
     * @throws \Throwable
     */
    protected function reportCreator(LogEvent $logEvent)
    {
        try
        {
            /** @var ReportCreator $creator */
            $creator = $this->app()->service('XF:Report\Creator', 'multiple_account', $logEvent->User);
            $message = $this->buildMessageForReport($logEvent);
            if ($message === null)
            {
                return null;
            }
            $creator->setMessage($message);

            if ($creator->validate($errors))
            {
                /** @var \XF\Mvc\Entity\Entity $reportOrThread */
                try
                {
                    $reportOrThread = $creator->save();
                }
                /** @noinspection PhpRedundantCatchClauseInspection */
                catch (\XF\Db\DuplicateKeyException $e)
                {
                    $reportOrThread = \XF::app()->find('XF:Report', ['multiple_account', $logEvent->triggering_user_id]);
                }
                if ($reportOrThread === null)
                {
                    if (\XF::$developmentMode)
                    {
                        \XF::logError('Unexpected null return result from ' . ReportCreator::class);
                    }
                    return null;
                }
                \XF::runLater(function() use ($reportOrThread, $creator) {
                    if ($reportOrThread->exists())
                    {
                        $creator->sendNotifications();
                    }
                });

                /** @noinspection PhpStatementHasEmptyBodyInspection */
                if ($reportOrThread instanceof \XF\Entity\Thread)
                {
                    // nothing to-do
                }
                else
                {
                    /** @noinspection PhpStatementHasEmptyBodyInspection */
                    if ($reportOrThread instanceof \XF\Entity\Report)
                    {
                        // nothing to-do
                    }
                    else if (\XF::$developmentMode)
                    {
                        throw new \LogicException('Unexpected reportable content type ' . get_class($reportOrThread) . ' returned from' . ReportCreator::class);
                    }
                    else
                    {
                        return null;
                    }
                }

                return $reportOrThread;
            }
            else
            {
                \XF::logException(new PrintableException($errors), false, 'Failed to create multi-account report:');
            }

            return null;
        }
        catch (\Throwable $e)
        {
            \XF::logException($e, true);
            if (\XF::$developmentMode)
            {
                throw $e;
            }
        }

        return null;
    }

    /**
     * @param LogEvent $logEvent
     * @return string|Phrase
     */
    protected function buildMessageForReport(LogEvent $logEvent)
    {
        $bbCode = '[multi_account_block]' . \json_encode([
                'event'         => $logEvent->event_id
            ]) . '[/multi_account_block]';

        if (\XF::options()->svSockSignupIncludeRawInfo ?? false)
        {
            $bbCode .= "\n" . \trim(\XF::app()->templater()->renderMacro('public:sv_multiple_account_macros', 'renderBbCode', [
                'event'         => $logEvent,
                'primaryUser'   => $logEvent->User,
                'primaryUserId' => $logEvent->triggering_user_id,
            ]));
        }

        return $bbCode;
    }

    /**
     * @param LogEvent $logEvent
     * @param int $reportDataId
     * @return bool
     */
    protected function hasIgnoredUserSetForReportEvent(LogEvent $logEvent, $reportDataId)
    {
        // check to see if this report_date_id has ignored records
        // at this stage $logEvent should not have been attached to a ReportData entity
        $rawPossibleIgnores = \XF::db()->fetchRow("
            SELECT DISTINCT `event`.triggering_user_id, `log`.user_id, `log`.detection_methods
            FROM xf_sv_multiple_account_log AS `log`
            JOIN xf_sv_multiple_account_event AS `event` ON `event`.event_id = `log`.event_id
            WHERE `log`.active IS NULL
                  AND `event`.report_data_id = ?
                  AND `event`.event_id <> ?
            ORDER BY `event`.detection_date ASC, `event`.triggering_user_id ASC, `log`.user_id ASC
        ", [$reportDataId, $logEvent->event_id]);
        if (!$rawPossibleIgnores)
        {
            return false;
        }

        // build the map of logs to compare to
        $logs = [];
        foreach($logEvent->Logs as $log)
        {
            if (!isset($logs[$log->user_id]))
            {
                $logs[$log->user_id] = $log;
            }
        }
        if (!$logs)
        {
            return false;
        }

        // decompose from a tuple into a user-id => detection methods map
        $possibleIgnores = [];
        foreach ($rawPossibleIgnores as $rawPossibleIgnore)
        {
            $detectionMethods = @\json_decode($rawPossibleIgnore['detection_methods'], true);
            if (!\is_array($detectionMethods))
            {
                continue;
            }
            $userId = $rawPossibleIgnore['triggering_user_id'];
            if ($userId && !isset($possibleIgnores[$userId]))
            {
                $possibleIgnores[$userId] = $detectionMethods;
            }
            $userId = $rawPossibleIgnore['user_id'];
            if ($userId && !isset($possibleIgnores[$userId]))
            {
                $possibleIgnores[$userId] = $detectionMethods;
            }
        }

        foreach($possibleIgnores as $userId => $detectionMethods)
        {
            // check if this user id + detection methods is known
            if (empty($logs[$userId]))
            {
                continue;
            }
            $log = $logs[$userId];

            $newDetection = $log->detection_methods;
            if (!\is_array($detectionMethods))
            {
                continue;
            }

            if ($detectionMethods === $newDetection)
            {
                return true;
            }

            foreach($newDetection as $type => $data)
            {
                if (!isset($detectionMethods[$type]))
                {
                    continue;
                }

                if ($this->compareDetectionMethod($type, $data, $detectionMethods[$type]))
                {
                    return true;
                }
            }
        }

        return false;
    }

    /**
     * @param string $type
     * @param array $a
     * @param array $b
     * @return bool
     */
    protected function compareDetectionMethod($type, $a, $b)
    {
        switch($type)
        {
            case 'ip':
                if (!isset($a['ip']) || !isset($b['ip']))
                {
                    return false;
                }
                $ip1 = \XF\Util\Ip::convertIpStringToBinary($a['ip']);
                $ip2 = \XF\Util\Ip::convertIpStringToBinary($b['ip']);
                return  $ip1 && $ip2 && $ip1 === $ip2;
            case 'cookie':
                if (!isset($a['expected']) || !isset($a['received']))
                {
                    return false;
                }
                if (!isset($b['expected']) || !isset($b['received']))
                {
                    return false;
                }
                return $a['expected'] === $b['expected'] && $a['received'] === $b['received'] ||
                       $a['expected'] === $b['received'] && $a['received'] === $b['expected'];
            default:
                return false;
        }
    }

    /**
     * @param int $reportDataId
     * @param int $skipEvent
     * @return int[]
     */
    protected function getExistingUsers($reportDataId, $skipEvent)
    {
        return \XF::db()->fetchAllColumn("
            SELECT DISTINCT `event`.triggering_user_id AS `user_id`
            FROM xf_sv_multiple_account_event AS `event`
            WHERE `event`.report_data_id = ? AND `event`.event_id <> ?
            UNION
            SELECT DISTINCT `log`.user_id 
            FROM xf_sv_multiple_account_event AS `event`
            JOIN xf_sv_multiple_account_log AS `log` ON `log`.event_id = `event`.event_id -- and `log`.active = 1
            WHERE `event`.report_data_id = ? AND `event`.event_id <> ?
        ", [$reportDataId, $skipEvent, $reportDataId, $skipEvent]);
    }

    /**
     * @param array    $justCreated
     * @param LogEvent $logEvent
     * @throws \Throwable
     */
    protected function bumpReportEntries($justCreated, LogEvent $logEvent)
    {
        $reportData = $logEvent->ReportData;
        $options = \XF::options();
        $forceThreadBump = (bool)($options->reportIntoForumId ?? 0);
        if (empty($justCreated['report']) && $reportData->Report && ($options->svSockSignupCheckReport ?? false) && !$forceThreadBump)
        {
            $reason = null;
            $reportState = $reportData->Report->report_state;
            // check the report state filter
            $sendDuplicate = $options->svSockSignupCheckDupeReports ?? [];
            $canBump = $sendDuplicate[$reportState] ?? false;
            if ($canBump)
            {
                $reason = \XF::phrase('sv_multi_account_report_reopen.report_state', ['state' => $reportState]);
            }
            if (!$canBump)
            {
                // if the last report state doesn't match the last assigned state, just bump anyway...
                $lastStateChange = \XF::db()->fetchOne(" 
                    SELECT state_change 
                    FROM xf_report_comment
                    WHERE report_id = ? AND state_change NOT IN ('','moved')
                    ORDER BY comment_date DESC
                    LIMIT 1
                ", $reportData->report_id);
                if ($lastStateChange && $lastStateChange != $reportState)
                {
                    $canBump = !empty($sendDuplicate[$lastStateChange]);
                    if ($canBump)
                    {
                        $reason = \XF::phrase('sv_multi_account_report_reopen.report_state_last', ['state' => $lastStateChange]);
                    }
                }
            }
            if (!$canBump)
            {
                // check if this is a multi-account event with new users
                $userIds = $this->getExistingUsers($logEvent->ReportData->report_data_id, $logEvent->event_id);
                $userIds = \array_fill_keys($userIds, true);
                if (empty($userIds[$logEvent->triggering_user_id]))
                {
                    $canBump = true;
                    $reason = \XF::phrase('sv_multi_account_report_reopen.unknown_user', ['userId' => $logEvent->triggering_user_id]);
                }
                else
                {
                    foreach ($logEvent->Logs as $log)
                    {
                        if (empty($userIds[$log->user_id]))
                        {
                            $canBump = true;
                            $reason = \XF::phrase('sv_multi_account_report_reopen.unknown_user', ['userId' => $log->user_id]);
                            break;
                        }
                    }
                }
            }
            if ($canBump)
            {
                $this->reportCommentCreator($reportData->Report, $logEvent, $reason);
            }
        }
    }

    /**
     * @param Report   $report
     * @param LogEvent $logEvent
     * @param Phrase   $bumpReason
     * @return ReportComment|null
     * @throws \Throwable
     */
    protected function reportCommentCreator(Report $report, LogEvent $logEvent, Phrase $bumpReason = null)
    {
        try
        {
            /** @var \XF\Service\Report\Commenter $creator */
            $creator = $this->app()->service('XF:Report\Commenter', $report);
            $message = $this->buildMessageForReport($logEvent);
            if ($message === null)
            {
                return null;
            }
            if ($bumpReason && (\XF::options()->svLogReportReopenReason ?? false))
            {
                $message = $message . "\n" . $bumpReason->render('raw');
            }
            $creator->setMessage($message);
            // make sure the report_data_id is included
            $contentInfo = $report->content_info;
            $contentInfo['report_data_id'] = $logEvent->report_data_id;
            $report->content_info = $contentInfo;
            if ($creator->validate($errors))
            {
                /** @var ReportComment $reportComment */
                $reportComment = $creator->save();

                \XF::runLater(function() use ($reportComment, $creator) {
                    if ($reportComment->exists())
                    {
                        $creator->sendNotifications();
                    }
                });

                return $reportComment;
            }
            else
            {
                \XF::logException(new PrintableException($errors), false, 'Failed to create multi-account report:');
            }

            return null;
        }
        catch (\Throwable $e)
        {
            \XF::logException($e, true);
            if (\XF::$developmentMode)
            {
                throw $e;
            }
        }

        return null;
    }

    /**
     * Detects multiple accounts for the $currentUser, $currentUser MAY NOT be fully constructed!
     *
     * @param User        $currentUser
     * @param string|null $receivedToken
     * @return DetectionRecord[]
     */
    public function detectMultipleAccounts(User $currentUser, $receivedToken)
    {
        /** @var int|null $currentUserId */
        $currentUserId = $currentUser->user_id;
        /** @var DetectionRecord[] $dataCollection */
        $dataCollection = [];
        $options = $this->options();
        $matchingMode = (int)($options->svSockSignupCheckMatchingMode ?? 0);
        $expectedToken = $currentUser->AccountDetectionToken->token;
        /** @var TokenEntity $tokenEntity */
        $tokenEntity = $receivedToken ? $this->getTokenFromCookie($receivedToken) : null;

        if ($tokenEntity && $receivedToken !== $expectedToken)
        {
            /** @var User $tokenOwner */
            $tokenOwner = $tokenEntity->User;

            if ($tokenOwner !== null)
            {
                $userId = $tokenOwner->user_id;
                // avoid reporting a multi-account event if the token is just being upgraded
                if ($userId && $userId !== $currentUserId)
                {
                    $dataCollection[$userId] = new DetectionRecord($tokenOwner, ['cookie' => new DetectionMethodCookie($expectedToken ?? '', $receivedToken ?? '')], $tokenEntity);
                }
            }
            else
            {
                // trigger setting a new cookie as the old account was deleted
                $receivedToken = null;
            }
        }
        if (!$tokenEntity)
        {
            $tokenEntity = $currentUser->AccountDetectionToken;
        }

        $ipOption = $options->svSockSignupCheckMatchingIps ?? [];
        $ipAddress = $this->app()->request()->getIp();
        if ($ipAddress && isset($ipOption['checkIp']) && $ipOption['checkIp'] && $ipOption['minTime'] && !$this->isIpAddressAllowed($ipAddress))
        {
            $users = $this->getUsersWithIp($ipAddress, (int)$ipOption['minTime']);
            if ($users)
            {
                if (!$tokenEntity)
                {
                    $tokenEntity = $this->getTokenForUser(reset($users));
                }
                $tokenUserId = $tokenEntity->user_id;

                /** @var User $user */
                foreach ($users as $user)
                {
                    $userId = $user->user_id;
                    if ($userId === $currentUserId)
                    {
                        continue;
                    }

                    if ($matchingMode === self::MATCHING_MODE_COOKIE_ONLY && $tokenUserId !== $userId)
                    {
                        continue;
                    }

                    if (isset($dataCollection[$user->user_id]))
                    {
                        /** @var  $data */
                        $data = $dataCollection[$user->user_id];
                        if (isset($data->methods['ip']))
                        {
                            /** @var DetectionMethodIp $ipMethod2 */
                            $ipMethod2 = $data->methods['ip'];
                            $ipMethod2->addIps($ipAddress);
                        }
                        else
                        {
                            $data->methods['ip'] = new DetectionMethodIp($ipAddress);
                        }
                    }
                    else
                    {
                        $dataCollection[$user->user_id] = new DetectionRecord($user, ['ip' => new DetectionMethodIp($ipAddress)]);
                    }
                }
            }
        }

        if (!$receivedToken || $expectedToken !== $receivedToken)
        {
            $this->setCookieValue($expectedToken);
        }

        if ($dataCollection && $matchingMode === self::MATCHING_MODE_AND)
        {
            $detectionMethods = 1;
            if ($ipOption['checkIp'])
            {
                ++$detectionMethods;
            }

            if ($detectionMethods > 1)
            {
                $uniqueMethods = [];
                foreach ($dataCollection as $data)
                {
                    foreach (array_keys($data->methods) AS $methodType)
                    {
                        if ($data->user->canBypassMultipleAccountDetection($methodType))
                        {
                            continue;
                        }
                        if (!isset($uniqueMethods[$methodType]))
                        {
                            $uniqueMethods[$methodType] = true;
                        }
                    }
                }

                if (\count($uniqueMethods) !== $detectionMethods)
                {
                    $dataCollection = [];
                }
            }
        }

        return $dataCollection;
    }

    /**
     * @param string $cookie
     * @return null|TokenEntity|\XF\Mvc\Entity\Entity
     */
    public function getTokenFromCookie($cookie)
    {
        return $this->finder('SV\SignupAbuseBlocking:Token')
                    ->where('token', '=', $cookie)
                    ->order('active', 'desc')
                    ->with(['User', 'User.Profile'], false)
                    ->fetchOne();
    }

    /**
     * @param string|null $ip
     * @return bool
     */
    protected function isIpAddressAllowed($ip = null)
    {
        if ($ip === null)
        {
            $ip = $this->app()->request()->getIp();
        }
        if (!$ip)
        {
            return false;
        }

        $raw = $this->options()->svSockSignupCheckIpAllowedList ?? '';
        if (\strlen($raw) === 0)
        {
            return false;
        }
        $allowedIps = array_filter(array_map('trim', explode(',', preg_replace('/\s+/', ',', $raw))));
        if (empty($allowedIps))
        {
            return false;
        }

        $binaryIp = \XF\Util\Ip::convertIpStringToBinary($ip);
        if (!$binaryIp)
        {
            return false;
        }

        foreach ($allowedIps as $allowedIp)
        {
            $results = \XF\Util\Ip::parseIpRangeString($allowedIp);
            if ($results && Ip::ipMatchesRange($binaryIp, $results['startRange'], $results['endRange']))
            {
                return true;
            }
        }

        return false;
    }

    /**
     * @param string $ip
     * @param int    $timeLimit
     * @return User[]|\XF\Mvc\Entity\Entity[]
     */
    public function getUsersWithIp(string $ip, int $timeLimit)
    {
        if (!$ip)
        {
            return [];
        }
        if ($timeLimit <= 0)
        {
            return [];
        }

        $ip = \XF\Util\Ip::convertIpStringToBinary($ip);
        if (!$ip)
        {
            return [];
        }

        $userIds = $this->db()->fetchAllColumn('
            SELECT user_id
            FROM xf_ip AS ip
            WHERE ip.ip = ? AND log_date >= ?
            GROUP BY ip.user_id
        ', [$ip, \XF::$time - $timeLimit * 60]);

        return \XF::finder('XF:User')
                  ->with('PermissionCombination')
                  ->whereIds($userIds)
                  ->with('Profile')
                  ->fetch()
                  ->toArray();
    }

    /**
     * @param User $user
     * @return null|TokenEntity
     */
    public function getTokenForUser(User $user)
    {
        if (!$user->user_id)
        {
            return null;
        }

        return $user->AccountDetectionToken;
    }

    /**
     * @param string|null $detectionAction
     * @return null|string
     * @throws PrintableException
     */
    public function getCookieValue($detectionAction)
    {
        $options = \XF::options();

        $cookieName = $options->svSockSignupAccountCookie ?? '';
        if (\strlen($cookieName) === 0)
        {
            return null;
        }
        // explicitly use $_COOKIE/setcookie super-global to bypass XF cookie handling
        $cookieValue = $_COOKIE[$cookieName] ?? '';
        // detect known-bad value and zap it
        if ($cookieValue)
        {
            if (\strpos($cookieValue, 'SV\SignupAbuseBlocking:Token[') === 0)
            {
                setcookie($cookieName, '', \XF::$time - 3600, '', '');
                setcookie($cookieName, '', \XF::$time - 3600, '/', '');
                $_COOKIE[$cookieName] = '';
            }
            else if (\preg_match_all('#' . preg_quote($cookieName) . '=SV%5CSignupAbuseBlocking%3AToken%5B;?#', $_SERVER['HTTP_COOKIE'] ?? '', $matches) && $matches)
            {
                setcookie($cookieName, '', \XF::$time - 3600, '/', '');
            }
        }
        /** @var User $visitor */
        $visitor = \XF::visitor();

        $cookieFormat = $options->svSockSignupCookieFormat ?? [];
        if (!empty($cookieFormat['legacy1']) || !empty($cookieFormat['legacy2']))
        {
            if (\strpos($cookieValue, ',') !== false || is_numeric($cookieValue))
            {
                $reportDataId = null;
                $cookieValue = $this->handleLegacyCookieValue($visitor, $cookieValue, $detectionAction, !empty($cookieFormat['legacy2']), $reportDataId);
                $this->setCookieValue($cookieValue);
            }
        }

        return $cookieValue;
    }

    /**
     * @param User|\XF\Entity\User $visitor
     * @param User|\XF\Entity\User $user
     * @return void
     * @throws PrintableException
     */
    public function handleSharedEmailLink(\XF\Entity\User $visitor, \XF\Entity\User $user)
    {
        $detectionMethods = ['emailLink' => new DetectionMethodEmailLink()];
        $db = $this->db();
        $db->beginTransaction();

        /** @var ReportData $reportData */
        $reportData = $this->app()->em()->create('SV\SignupAbuseBlocking:ReportData');
        $reportData->save(true, false);

        /** @var LogEvent $logEvent */
        $logEvent = $this->app()->em()->create('SV\SignupAbuseBlocking:LogEvent');
        $logEvent->detection_action = 'emailLink';
        $logEvent->triggering_user_id = $visitor->user_id;
        $logEvent->username = $visitor->username;
        $logEvent->hydrateRelation('User', $visitor);
        $logEvent->report_data_id = $reportData->report_data_id;

        $logEvent->createLog($detectionMethods, $user, $user->AccountDetectionToken);

        $logEvent->save(true, false);
        $reportData->saveIfChanged($saved, true, false);

        $db->commit();
    }

    /**
     * @param User|\XF\Entity\User $visitor
     * @param string|string[]      $cookieValue
     * @param string               $detectionAction
     * @param bool                 $allowMultiple
     * @param int|null             $reportDataId
     * @return string
     * @throws PrintableException
     */
    public function handleLegacyCookieValue(\XF\Entity\User $visitor, $cookieValue, $detectionAction, $allowMultiple, &$reportDataId)
    {
        if (\is_array($cookieValue))
        {
            $userIds = $cookieValue;
            $cookieValue = implode(',', $cookieValue);
        }
        else if ($allowMultiple && \strpos($cookieValue, ',') !== false)
        {
            $userIds = \explode(',', $cookieValue);
            $userIds = \array_map('\intval', $userIds);
            $userIds = \array_unique($userIds, SORT_NUMERIC);
            $userIds = \array_fill_keys($userIds, null);
        }
        else
        {
            $userIds = [(int)$cookieValue => null];
        }
        unset($userIds[0]);
        unset($userIds[$visitor->user_id]);
        $detectionMethods = ['legacy' => new DetectionMethodLegacy($cookieValue ?? '')];

        $token = null;

        if ($userIds)
        {
            $users = $this->finder('XF:User')
                          ->whereIds(\array_keys($userIds))
                          ->fetch()->toArray();

            $db = $this->db();
            $db->beginTransaction();

            // require a mostly empty $reportData for later multiple account detection events to find this data

            /** @var ReportData $reportData */
            $reportData = \XF::app()->find('SV\SignupAbuseBlocking:ReportData', $reportDataId);
            if (!$reportData)
            {
                /** @var ReportData $reportData */
                $reportData = $this->app()->em()->create('SV\SignupAbuseBlocking:ReportData');
                $reportData->save(true, false);
                $reportDataId = $reportData->report_data_id;
            }

            /** @var LogEvent $logEvent */
            $logEvent = $this->app()->em()->create('SV\SignupAbuseBlocking:LogEvent');
            $logEvent->detection_action = $detectionAction;
            $logEvent->triggering_user_id = $visitor->user_id;
            $logEvent->username = $visitor->username;
            $logEvent->hydrateRelation('User', $visitor);
            $logEvent->report_data_id = $reportData->report_data_id;

            foreach ($userIds AS $userId => $username)
            {
                /** @var User $user */
                $user = $users[$userId] ?? null;
                if ($user)
                {
                    $username =  $user->username;
                }
                else
                {
                    $username = $username ?: ''. $userId;
                }
                if ($user)
                {
                    // ensure this user has a token
                    $userToken = $user->AccountDetectionToken;
                    if (empty($token))
                    {
                        $token = $userToken;
                    }
                }

                $logEvent->createLog($detectionMethods, $user, $token, $userId, $username);
            }

            $logEvent->save(true, false);
            $reportData->saveIfChanged($saved, true, false);

            $db->commit();
        }

        if (!$token)
        {
            /** @var TokenEntity $token */
            $token = $visitor->AccountDetectionToken;
        }

        return $token->token;
    }

    /**
     * Sets the multiple account cookie value.
     *
     * @param TokenEntity|string|null $value The cookie. Falsy to remove cookie.
     * @param int            $time  int How long the cookie is valid for, in seconds.
     */
    public function setCookieValue($value, $time = null)
    {
        $options = \XF::options();
        if (!($options->svSockSignupCheckCookie ?? false))
        {
            return;
        }

        // this will be done at the end of the registration request, to avoid over-stamping the value due to rejection
        $duringRegistration = Globals::$duringRegistration ?? false;
        if ($duringRegistration)
        {
            return;
        }
        if ($value instanceof TokenEntity)
        {
            $value = $value->token;
        }

        if ($time === null)
        {
            $time = ($this->options()->svSockSignupAccountCookieLifeSpan ?? 24) * 2592000;
        }

        $cookieName = $this->options()->svSockSignupAccountCookie ?? '';
        if (\strlen($cookieName) === 0)
        {
            return;
        }
        $expire = !$value ? \XF::$time - 3600 : \XF::$time + $time;
        // explicitly use $_COOKIE/setcookie super-global to bypass XF cookie handling, changing this will break cookie path magic!
        // path = "", domain = "", matches to the current domain + current path (or any sub-path)
        \setcookie($cookieName, $value, $expire, "", "");//, false, false);
        // ensure calling setCookieValue => getCookieValue, works as expected
        $_COOKIE[$cookieName] = $value;
    }
}
