<?php

namespace SV\SignupAbuseBlocking\Entity;

use XF\Mvc\Entity\Entity;
use XF\Mvc\Entity\Structure;

/**
 * COLUMNS
 * @property int|null                                 report_data_id
 * @property int|null                                 report_id
 * @property int                                      log_count
 * @property bool                                     active
 *
 * RELATIONS
 * @property-read \SV\SignupAbuseBlocking\XF\Entity\Report Report
 * @property-read LogEvent[]                               Events
 */
class ReportData extends Entity
{
    public function countUniqueAccounts(): int
    {
        /** @var \SV\SignupAbuseBlocking\Finder\Log $finder */
        $finder = $this->finder('SV\SignupAbuseBlocking:Log');
        $finder->inReportData($this->report_data_id)->asActive();
        $logUsers = $finder->fetchColumns(['LogEvent.triggering_user_id', 'user_id', 'LogEvent.username', 'username' => 'logUsername']);

        $userIds = [];
        foreach ($logUsers as $row)
        {
            if ($row['triggering_user_id'])
            {
                $userIds[$row['triggering_user_id']] = true;
            }
            else if ($row['username'])
            {
                $userIds[$row['username']] = true;
            }
            if ($row['user_id'])
            {
                $userIds[$row['user_id']] = true;
            }
            else if ($row['logUsername'])
            {
                $userIds[$row['logUsername']] = true;
            }
        }

        return \max(\count($userIds) - 1, 0);
    }

    public function getGroupedLogs(): array
    {
        /** @var \SV\SignupAbuseBlocking\Finder\Log $finder */
        $finder = $this->finder('SV\SignupAbuseBlocking:Log');
        $finder->inReportData($this->report_data_id)->asActive();
        $logs = $finder->fetch();
        $finder->loadUserRecords($logs);

        /** @var Log $log */
        $log = $logs->first();
        $primaryUser = $log ? $log->LogEvent->User : null;
        $logsByUser = [];
        $users = [];

        $userId = $log->LogEvent->triggering_user_id ?: 'guest-' . $log->LogEvent->username;
        $users[$userId] = $primaryUser;
        $logsByUser[$userId]['Event'][$log->LogEvent->event_id] = $log->LogEvent;

        foreach ($logs as $log)
        {
            $token = $log->LogEvent->triggering_user_id ?: 'guest-' . $log->LogEvent->username;
            $users[$token] = $log->LogEvent->User;
            if (!$primaryUser)
            {
                $primaryUser = $log->LogEvent->User;
            }

            $userId = $log->user_id ?: 'guest-' . $log->username;
            $users[$userId] = $log->User;

            if (!isset($logsByUser[$userId]['Event']))
            {
                $logsByUser[$userId]['Event'] = [];
            }
            $logsByUser[$userId]['Event'][$log->LogEvent->event_id] = $log->LogEvent;

            if (!isset($logsByUser[$userId]['UserLog']))
            {
                $logsByUser[$userId]['UserLog'] = [];
            }
            $logsByUser[$userId]['UserLog'][$log->log_id] = $log;
        }

        return [
            'reportDataId'  => $this->report_data_id,
            'primaryUser'   => $primaryUser,
            'primaryUserId' => $primaryUser ? $primaryUser->user_id : null,
            'users'         => $users,
            'logsByUser'    => $logsByUser,
        ];
    }

    /** @noinspection PhpUnusedParameterInspection */
    public function logAdded(Log $log)
    {
        $this->log_count++;
    }

    /** @noinspection PhpUnusedParameterInspection */
    public function logMadeActive(Log $log)
    {
        $this->log_count++;
    }

    /** @noinspection PhpUnusedParameterInspection */
    public function logMadeInactive(Log $log)
    {
        $this->log_count--;
    }

    /** @noinspection PhpUnusedParameterInspection */
    public function logRemoved(Log $log)
    {
        $this->log_count--;
    }

    protected function _preSave()
    {
        if (!$this->active)
        {
            $this->active = null;
        }
    }

    public static function getStructure(Structure $structure): Structure
    {
        $structure->table = 'xf_sv_multiple_account_report_data';
        $structure->shortName = 'SV\SignupAbuseBlocking:ReportData';
        $structure->primaryKey = 'report_data_id';
        $structure->columns = [
            'report_data_id' => ['type' => self::UINT, 'autoIncrement' => true, 'nullable' => true],
            'report_id'      => ['type' => self::UINT, 'nullable' => true, 'default' => null],
            'log_count'      => ['type' => self::UINT, 'default' => 0],
            'active'         => ['type' => self::BOOL, 'nullable' => true, 'default' => true],
        ];

        $structure->relations = [
            'Report' => [
                'entity'     => 'XF:Report',
                'type'       => self::TO_ONE,
                'conditions' => 'report_id',
                'primary'    => true
            ],
            'Events' => [
                'entity'     => 'XF:Report',
                'type'       => self::TO_MANY,
                'conditions' => 'report_data_id',
                'primary'    => true
            ],
        ];

        $structure->defaultWith = [];

        return $structure;
    }
}