<?php
/**
 * @noinspection PhpMissingReturnTypeInspection
 */

namespace SV\SignupAbuseBlocking\XF\ControllerPlugin;

use SV\SignupAbuseBlocking\Entity\Token;
use SV\SignupAbuseBlocking\Globals;
use SV\SignupAbuseBlocking\MultipleAccount\DetectionRecord;

class Login extends XFCP_Login
{
    /** @var bool */
    protected $doMultiAccountCheck = true;

    protected function clearCookieSkipList()
    {
        $cookies = parent::clearCookieSkipList();
        if ($cookieName = \XF::options()->svSockSignupAccountCookie)
        {
            $cookies[] = $cookieName;
        }

        return $cookies;
    }

    public function logoutVisitor()
    {
        /** @var \SV\SignupAbuseBlocking\XF\Entity\User $visitor */
        $visitor = \XF::visitor();
        if ($visitor->user_id && !$this->skipMultiAccountCheck())
        {
            try
            {
                /** @var \SV\SignupAbuseBlocking\Repository\MultipleAccount $multipleAccountRepo */
                $multipleAccountRepo = $this->repository('SV\SignupAbuseBlocking:MultipleAccount');
                $receivedToken = $multipleAccountRepo->getCookieValue('logout');
                if (!$receivedToken)
                {
                    $multipleAccountRepo->setCookieValue($visitor->AccountDetectionToken);
                }
            }
            catch (\Throwable $e)
            {
                // do not block login if any sort of error occurs
                \XF::logException($e, true);
                if (\XF::$developmentMode)
                {
                    throw $e;
                }
            }
        }

        parent::logoutVisitor();
    }

    protected function skipMultiAccountCheck()
    {
        if (!(Globals::$armLoginPlugin ?? false))
        {
            return true;
        }

        // support https://xenforo.com/community/threads/login-as-user-lau2.153010/
        $controller = $this->controller;
        if (!empty($controller->lau_login))
        {
            return true;
        }

        return !$this->doMultiAccountCheck;
    }

    public function completeLogin(\XF\Entity\User $user, $remember)
    {
        parent::completeLogin($user, $remember);

        if ($this->skipMultiAccountCheck())
        {
            return;
        }

        /** @var \SV\SignupAbuseBlocking\XF\Entity\User $visitor */
        $visitor = \XF::visitor();
        if (!$visitor->user_id)
        {
            return;
        }

        try
        {
            /** @var \SV\SignupAbuseBlocking\Repository\MultipleAccount $multipleAccountRepo */
            $multipleAccountRepo = $this->repository('SV\SignupAbuseBlocking:MultipleAccount');
            $receivedToken = $multipleAccountRepo->getCookieValue('login');

            /**
             * @var DetectionRecord[] $detectMethods
             * @var Token             $token
             */
            $dataCollection = $multipleAccountRepo->detectMultipleAccounts($visitor, $receivedToken);
            if ($dataCollection)
            {
                $multipleAccountRepo->processMultipleAccountDetection($visitor, $dataCollection, 'login');
            }
        }
        catch (\Throwable $e)
        {
            // do not block login if any sort of error occurs
            \XF::logException($e, true);
            if (\XF::$developmentMode)
            {
                throw $e;
            }
        }
    }

    public function actionKeepAlive()
    {
        $response =  parent::actionKeepAlive();

        try
        {
            /** @var \SV\SignupAbuseBlocking\XF\Entity\User $visitor */
            $visitor = \XF::visitor();
            if (!$visitor->user_id)
            {
                return $response;
            }

            if ($this->skipMultiAccountCheck())
            {
                return $response;
            }

            /** @var \SV\SignupAbuseBlocking\Repository\MultipleAccount $multipleAccountRepo */
            $multipleAccountRepo = $this->repository('SV\SignupAbuseBlocking:MultipleAccount');
            $receivedToken = $multipleAccountRepo->getCookieValue('login');

            if (!$receivedToken)
            {
                // no cookie, set it
                $multipleAccountRepo->setCookieValue($visitor->AccountDetectionToken);
            }
        }
        catch (\Throwable $e)
        {
            // do not block login if any sort of error occurs
            \XF::logException($e, true);
            if (\XF::$developmentMode)
            {
                throw $e;
            }
        }

        return $response;
    }
}