<?php

namespace SV\SignupAbuseBlocking\Spam\Checker\User;

use SV\SignupAbuseBlocking\Globals;
use SV\SignupAbuseBlocking\MultipleAccount\DetectionMethod;
use SV\SignupAbuseBlocking\MultipleAccount\DetectionRecord;
use SV\SignupAbuseBlocking\Spam\IScoringChecker;
use XF\Entity\User;
use XF\Spam\Checker\AbstractProvider;
use XF\Spam\Checker\UserCheckerInterface;

class MultipleAccount extends AbstractProvider implements UserCheckerInterface
{
    protected function getType(): string
    {
        return 'SignupAbuseMultiAccount';
    }

    /**
     * @param User  $user
     * @param array $extraParams
     * @throws \Throwable
     * @throws \XF\PrintableException
     */
    public function check(User $user, array $extraParams = [])
    {
        if (!($this->checker instanceof IScoringChecker))
        {
            // not configured
            return;
        }
        /** @var \SV\SignupAbuseBlocking\XF\Entity\User $user */
        /** @var \SV\SignupAbuseBlocking\Repository\MultipleAccount $multipleAccountRepo */
        $multipleAccountRepo = $this->app()->repository('SV\SignupAbuseBlocking:MultipleAccount');
        $token = $multipleAccountRepo->getCookieValue('register');

        Globals::$dataCollection = null;
        try
        {
            $dataCollection = $multipleAccountRepo->detectMultipleAccounts($user, $token);
            $this->doCheck($this->checker, $dataCollection);
            Globals::$dataCollection = $dataCollection;
        }
        catch (\Throwable $e)
        {
            // do not block login if any sort of error occurs
            \XF::logException($e, true);
            if (\XF::$developmentMode)
            {
                throw $e;
            }
        }
    }

    /**
     * @param IScoringChecker   $checker
     * @param DetectionRecord[] $dataCollection
     */
    protected function doCheck(IScoringChecker $checker, array $dataCollection = null)
    {
        $options = \XF::options();

        $registrationMode = $options->svSockSignupCheckRegMode ?? '';
        $registrationModeForBanned = $options->svSockSignupCheckRegModeGroup ?? '';
        $specialGroupIds = $options->svSockSignupCheckRegModeGroupIds ?? '';
        if ($specialGroupIds)
        {
            $specialGroupIds = array_map('\intval', explode(",", $specialGroupIds));
        }
        else
        {
            $specialGroupIds = [];
        }

        $action = 'allowed';
        if (!$dataCollection)
        {
            $this->logDecision($action);

            return;
        }

        $typeId = 0;
        foreach ($dataCollection as $data)
        {
            $typeId++;
            $actionToTake = $registrationMode;

            /**
             * @var string          $detectionMethod
             * @var DetectionMethod $detectionData
             */
            $methods = [];
            foreach ($data->methods as $detectionMethod => $detectionData)
            {
                if ($data->user->canBypassMultipleAccountDetection($detectionMethod))
                {
                    continue;
                }
                $methods[] = \XF::phrase('sv_multiple_account_method.' . $detectionMethod, $detectionData->forPhraseData())->render('raw');
            }
            if (!$methods)
            {
                continue;
            }
            $methods = implode(", ", $methods);

            $phraseData = [
                'method'   => $methods,
                'username' => $data->user->username,
                'user_id'  => $data->user->user_id,
            ];

            if ($data->user->is_banned)
            {
                $actionToTake = $registrationModeForBanned;
                $this->logDetail2($typeId, '_banned', 'sv_reg_log.multi_account_is_banned', $phraseData);
            }

            if ($registrationModeForBanned)
            {
                $groups = $data->user->secondary_group_ids;
                $groups[] = $data->user->user_group_id;
                $intersect = array_intersect($groups, $specialGroupIds);

                if ($intersect)
                {
                    $actionToTake = $registrationModeForBanned;

                    /** @var \XF\Entity\ChangeLog $changeLog */
                    $changeLog = \XF::em()->create('XF:ChangeLog');
                    $changeLog->content_type = 'user';
                    $changeLog->old_value = '';
                    $changeLog->new_value = implode(',', $intersect);
                    $changeLog->field = 'secondary_group_ids';
                    $changeLog->setReadOnly(true);

                    /** @var \XF\ChangeLog\DisplayEntry $displayLog */
                    $displayLog = $changeLog->getDisplayEntry();
                    $phraseData['groups'] = $displayLog->new;

                    $this->logDetail2($typeId, '_member', 'sv_reg_log.multi_account_group', $phraseData);
                }
            }

            switch (\strval($actionToTake))
            {
                case 'reject':
                    $checker->logScoreReject('sv_reg_log.multi_account_reject', $phraseData);
                    break;
                case 'moderate':
                    $checker->logScoreModerate('sv_reg_log.multi_account_moderate', $phraseData);
                    break;
                case '0':
                    $checker->logScoreAccept('sv_reg_log.multi_account_accept', $phraseData);
                    break;
                default:
                    // this really should be sv_reg_log.multi_account_fail, but to avoid needing to migrate data it is not renamed
                    $checker->logScore('sv_reg_log.multi_account_accept_score', $actionToTake, $phraseData);
                    break;
            }
        }

        $this->logDecision($action);
    }

    protected function logDetail2(int $typeId, string $typeSuffix, string $phrase, array $data = [])
    {
        $this->checker->logDetail($this->getType() . '.' . $typeId . $typeSuffix, $phrase, $data);
    }

    public function submit(User $user, array $extraParams = [])
    {
    }
}