<?php

namespace SV\SignupAbuseBlocking\Finder;

use XF\Mvc\Entity\AbstractCollection;
use XF\Mvc\Entity\Finder;
use XF\Entity\User;

class Log extends Finder
{
    /**
     * @param int|User $userId
     * @return self
     */
    public function asAlterUser($userId): self
    {
        if ($userId instanceof User)
        {
            $userId = $userId->user_id;
        }

        $this->where('user_id', '=', $userId);

        return $this;
    }

    /**
     * @param int|User $userId
     * @return self
     */
    public function byUser($userId): self
    {
        if ($userId instanceof User)
        {
            $userId = $userId->user_id;
        }

        $this->whereOr(['user_id', '=', $userId], ['LogEvent.triggering_user_id', '=', $userId]);

        return $this;
    }

    /**
     * @param bool $alertable
     * @return self
     */
    public function asAlertable(bool $alertable = true): self
    {
        $this->where('is_alertable', '=', $alertable);

        return $this;
    }

    /**
     * @param bool $active
     * @return self
     */
    public function asActive(bool $active = true): self
    {
        $this->where('active', '=', $active);

        return $this;
    }

    /**
     * @param int $reportDataId
     * @return self
     */
    public function inReportData(int $reportDataId): self
    {
        $this->with(['LogEvent'], true);

        $this->where('LogEvent.report_data_id', '=', $reportDataId);

        $this->order('LogEvent.detection_date', 'ASC');
        $this->order('LogEvent.triggering_user_id', 'ASC');
        $this->order('user_id', 'ASC');

        return $this;
    }

    public function loadUserRecords(AbstractCollection $logs)
    {
        $userIds = [];
        $userOptions = [];
        $userPrivacy = [];
        $reportDataIds = [];
        $em = $this->em;
        /** @var \SV\SignupAbuseBlocking\Entity\Log $log */
        foreach ($logs as $log)
        {
            $userId = $log->user_id;
            if ($userId)
            {
                if (!$em->findCached('XF:User', $userId))
                {
                    $userIds[$userId] = true;
                }
                else
                {
                    if (!$em->findCached('XF:UserOption', $userId))
                    {
                        $userOptions[$userId] = true;
                    }
                    if (!$em->findCached('XF:UserPrivacy', $userId))
                    {
                        $userPrivacy[$userId] = true;
                    }
                }
            }

            $logEvent = $log->LogEvent;
            if ($logEvent === null)
            {
                continue;
            }

            $userId = $logEvent->triggering_user_id;
            if ($userId && !$em->findCached('XF:User', $userId))
            {
                $userIds[$userId] = true;
            }

            $reportDataId = $logEvent->report_data_id;
            if ($reportDataId && !$em->findCached('SV\SignupAbuseBlocking:ReportData', $reportDataId))
            {
                $reportDataIds[$reportDataId] = true;
            }
        }

        if ($userIds)
        {
            \XF::finder('XF:User')->with(['Profile','Privacy'])->whereIds(\array_keys($userIds))->fetch();
        }
        if ($userOptions)
        {
            \XF::finder('XF:UserOption')->whereIds(\array_keys($userOptions))->fetch();
        }
        if ($userPrivacy)
        {
            \XF::finder('XF:UserPrivacy')->whereIds(\array_keys($userPrivacy))->fetch();
        }
        if ($reportDataIds)
        {
            \XF::finder('SV\SignupAbuseBlocking:ReportData')->whereIds(\array_keys($reportDataIds))->fetch();
        }

        $permCache = [];
        // hydrate nulls..
        /** @var \SV\SignupAbuseBlocking\Entity\Log $log */
        foreach ($logs as $log)
        {
            $userId = $log->user_id;
            if (!$userId || !$em->findCached('XF:User', $userId))
            {
                $log->hydrateRelation('User', null);
            }
            if ($log->User !== null)
            {
                $permCache[$log->User->permission_combination_id] = true;
            }

            $logEvent = $log->LogEvent;
            if ($logEvent !== null)
            {
                $userId = $logEvent->triggering_user_id;
                if (!$userId || !$em->findCached('XF:User', $userId))
                {
                    $logEvent->hydrateRelation('User', null);
                }
                if ($logEvent->User !== null)
                {
                    $permCache[$logEvent->User->permission_combination_id] = true;
                }
            }
        }

        if ($permCache)
        {
            /** @var \SV\CanWarnStaff\XF\Repository\User $userRepo */
            $userRepo = \XF::repository('XF:User');
            if (\is_callable([$userRepo, 'preloadGlobalPermissionsFromIds']))
            {
                $userRepo->preloadGlobalPermissionsFromIds(array_keys($permCache));
            }
        }
    }
}