<?php

namespace SV\SignupAbuseBlocking\Entity;

use SV\SignupAbuseBlocking\XF\Entity\User;
use XF\Mvc\Entity\AbstractCollection;
use XF\Mvc\Entity\Entity;
use XF\Mvc\Entity\Structure;

/**
 * COLUMNS
 *
 * @property int|null                 event_id
 * @property int                      detection_date
 * @property int                      report_data_id
 * @property int                      triggering_user_id
 * @property string                   username
 * @property int                      log_count
 * @property string                   detection_action
 * GETTERS
 * @property-read User                     User
 * @property-read string|\XF\Phrase        DetectionAction
 * RELATIONS
 * @property-read User                     User_
 * @property-read ReportData               ReportData
 * @property-read AbstractCollection|Log[] Logs
 */
class LogEvent extends Entity
{
    public function clearCascadeSaveHack()
    {
        $this->_cascadeSave = [];
    }

    public function createLog(array $detectionMethods, User $user = null, Token $token = null, int $userId = null, string $username = null): Log
    {
        $userId = $user->user_id ?? ($userId ?: 0);
        $username = $user->username ?? ($username ?: "Guest ({$userId})");
        if (!$detectionMethods)
        {
            \XF::logError('Unexpected lack of detection records:'.$userId . "-". $username );
        }

        $methods = [];
        foreach ($detectionMethods as $detectionMethod => $detectionData)
        {
            $methods[$detectionMethod] = $detectionData->forPhraseData();
        }

        /** @var Log $logEntity */
        $logEntity = $this->app()->em()->create('SV\SignupAbuseBlocking:Log');
        $logEntity->event_id = $this->exists() ? $this->event_id : $this->_getDeferredValue(function () { return $this->event_id; }, 'save');
        $logEntity->detection_methods = $methods;
        $logEntity->token_id = $token ? $token->token_id : null;

        if ($user)
        {
            $logEntity->user_id = $user->user_id;
            $logEntity->username = $user->username;
            if (!$user->Profile->multiple_account_detection_alertable)
            {
                $logEntity->is_alertable = false;
                $logEntity->active = false;
            }
        }
        else
        {
            $logEntity->user_id = $userId;
            $logEntity->username = $username;
        }

        $logEntity->hydrateRelation('LogEvent', $this);
        $logEntity->hydrateRelation('Token', $token);
        $logEntity->hydrateRelation('User', $user);

        $this->addCascadedSave($logEntity);

        return $logEntity;
    }

    /** @noinspection PhpUnusedParameterInspection */
    public function logAdded(Log $log)
    {
        $this->log_count++;
        if ($this->ReportData)
        {
            $this->ReportData->log_count++;
        }
    }

    /** @noinspection PhpUnusedParameterInspection */
    public function logMadeActive(Log $log)
    {
        $this->log_count++;
        if ($this->ReportData)
        {
            $this->ReportData->log_count++;
        }
    }

    /** @noinspection PhpUnusedParameterInspection */
    public function logMadeInactive(Log $log)
    {
        $this->log_count--;
        if ($this->ReportData)
        {
            $this->ReportData->log_count++;
        }
    }

    protected function _postSave()
    {
        $logCount = \intval($this->db()->fetchOne('
            select count(*)
            from xf_sv_multiple_account_log
            where event_id = ? and active = 1
        ', [$this->event_id]));

        $this->fastUpdate('log_count', $logCount);
        if ($this->ReportData)
        {
            $this->ReportData->fastUpdate('log_count', $this->ReportData->log_count + $logCount);
        }
    }

    protected function _postDelete()
    {
        throw new \LogicException("No Supported");
    }

    protected function getUser(): \XF\Entity\User
    {
        $user = $this->User_;

        if (!$user)
        {
            $username = $this->username;
            if (!$username)
            {
                $username = 'user id:' . $this->triggering_user_id;
            }
            /** @var \XF\Repository\User $userRepo */
            $userRepo = \XF::repository('XF:User');
            /** @var User $user */
            $user = $userRepo->getGuestUser($username);
            $user->svInitGuestUser(0, [
                'is_banned' => true,
            ]);
        }

        return $user;
    }

    /**
     * @return array|\XF\Phrase
     */
    protected function getDetectionAction()
    {
        return \XF::phrase('sv_multiple_account_action.' . $this->detection_action);
    }

    public static function getStructure(Structure $structure): Structure
    {
        $structure->table = 'xf_sv_multiple_account_event';
        $structure->shortName = 'SV\SignupAbuseBlocking:LogEvent';
        $structure->primaryKey = 'event_id';
        $structure->columns = [
            'event_id'           => ['type' => self::UINT, 'autoIncrement' => true, 'nullable' => true],
            'detection_date'     => ['type' => self::UINT, 'default' => \XF::$time],
            'report_data_id'     => ['type' => self::UINT, 'required' => false, 'nullable' => true],
            'triggering_user_id' => ['type' => self::UINT, 'required' => true],
            'username'           => ['type' => self::STR,  'maxLength' => 50, 'required' => true],
            'log_count'          => ['type' => self::UINT, 'default' => 0],
            'detection_action'   => ['type' => self::STR, 'required' => true],
        ];
        $structure->relations = [
            'User'       => [
                'entity'     => 'XF:User',
                'type'       => self::TO_ONE,
                'conditions' => [['user_id', '=', '$triggering_user_id']],
                'primary'    => true
            ],
            'ReportData' => [
                'entity'     => 'SV\SignupAbuseBlocking:ReportData',
                'type'       => self::TO_ONE,
                'conditions' => 'report_data_id',
                'primary'    => true
            ],
            'Logs'       => [
                'entity'     => 'SV\SignupAbuseBlocking:Log',
                'type'       => self::TO_MANY,
                'conditions' => 'event_id',
                'primary'    => true
            ],
        ];

        $structure->getters = [
            'User'             => ['getter' => 'getUser', 'cache' => true],
            'DetectionAction'  => ['getter' => 'getDetectionAction', 'cache' => true],
        ];

        return $structure;
    }
}