<?php
class ModEss_Model_Log extends XFCP_ModEss_Model_Log
{
    protected $_forumLinksCache = array();
    protected $_profileLinksCache = array();
    protected $_resourceLinksCache = array();
    protected $_resourceCategoryLinksCache = array();
    protected $_nodeIdsCache = array();

    public function getViewableModLogEntries($cutOff, array $fetchOptions = array(), $maxEntries = null)
    {
        $entries = $this->getModLogEntriesByDate($cutOff, $fetchOptions);
        if (!empty($entries))
            $entries = $this->_weedOutUnViewableEntries($entries, $maxEntries);

        return $entries;
    }

    public function getModLogEntriesByDate($cutOff, array $fetchOptions = array())
    {
        $db = $this->_getDb();
        $cutOff = intval($cutOff);
        $entries = array();

        if ($cutOff) // if no last log view then no known new entries
        {
            $excludeUserIds = null;
            $includeUserIds = null;
            $typesIn = $typesOut = array();
            $actionsIn = $skipActionsIn = $skipActionsOut = array();

            if (!empty($fetchOptions))
            {
                // skipped actions
                if (isset($fetchOptions['skip_actions']) && is_array($fetchOptions['skip_actions']))
                {
                    $skipActions = $fetchOptions['skip_actions'];
                    foreach ($skipActions AS $inOut => $skip)
                    {
                        foreach ($skip AS $type => $actions)
                        {
                            if ($inOut == 'include') // type IN () AND action IN ()
                            {
                                foreach ($actions AS $action)
                                {
                                    $skipActionsIn[$action] = trim(str_replace("'", "", $action));
                                }
                            }
                            else // 'exclude' // type NOT IN () OR action NOT IN ()
                            {
                                $typesOut[$type] = $type;

                                foreach ($actions AS $action)
                                {
                                    $skipActionsOut[$action] = trim(str_replace("'", "", $action));
                                }
                            }
                        }
                    }
                }

                // included actions
                if (isset($fetchOptions['actions']) && is_array($fetchOptions['actions']))
                {
                    $actions = $fetchOptions['actions'];
                    foreach ($actions AS $type => $actions)
                    {
                        $typesIn[$type] = $type;

                        foreach ($actions AS $action)
                        {
                            $actionsIn[$action] = trim(str_replace("'", "", $action));
                        }
                    }
                }

                if (isset($fetchOptions['content_type']) && $fetchOptions['content_type'] != 'all')
                {
                    $typesIn[$fetchOptions['content_type']] = $fetchOptions['content_type'];

                    if ($fetchOptions['content_type'] == 'resource')
                        $typesIn['resource_update'] = 'resource_update';
                }

                // exclude or include users
                if (!isset($fetchOptions['include_user_ids']) && isset($fetchOptions['exclude_user_ids']) && is_array($fetchOptions['exclude_user_ids']))
                {
                    $excludeUserIds = $fetchOptions['exclude_user_ids'];
                }
                else if (isset($fetchOptions['include_user_ids']) && is_array($fetchOptions['include_user_ids']))
                {
                    $includeUserIds = $fetchOptions['include_user_ids'];
                }
            }

            $whereClause = '';

            if (!empty($typesIn))
            {
                if (!empty($actionsIn))
                    $whereClause .= ' AND (moderator_log.content_type IN (' . $db->quote($typesIn) . ') AND moderator_log.action IN (' . $db->quote($actionsIn) . '))';
                else
                    $whereClause .= ' AND moderator_log.content_type IN (' . $db->quote($typesIn) . ')';
            }

            if (!empty($skipActionsIn))
            {
                $whereClause .= ' AND moderator_log.action NOT IN (' . $db->quote($skipActionsIn) . ')';
            }

            if (!empty($typesOut))
            {
                if (!empty($skipActionsOut))
                    $whereClause .= ' AND (moderator_log.content_type NOT IN (' . $db->quote($typesOut) . ') OR moderator_log.action NOT IN (' . $db->quote($skipActionsOut) . '))';
                else
                    $whereClause .= ' AND moderator_log.content_type NOT IN (' . $db->quote($typesOut) . ')';
            }

            if (isset($excludeUserIds))
            {
                $whereClause .= ' AND moderator_log.user_id NOT IN (' . $db->quote($excludeUserIds) . ')';
            }

            if (isset($includeUserIds))
            {
                $whereClause .= ' AND moderator_log.user_id IN (' . $db->quote($includeUserIds) . ')';
            }

            $entries = $this->fetchAllKeyed('
					SELECT moderator_log.*, user.username, user.avatar_date, user.avatar_width, user.avatar_height, user.gravatar
					FROM xf_moderator_log AS moderator_log
					INNER JOIN xf_user AS user ON (user.user_id = moderator_log.user_id)
					WHERE moderator_log.log_date > ?'
                . $whereClause . '
					ORDER BY moderator_log.log_date DESC
					LIMIT 500
			', 'moderator_log_id', $cutOff);
        }

        return $entries;
    }

    // in terms of whether user has the view Moderator Thread Log permission for the forum in which the item resides
    protected function _weedOutUnViewableEntries(array $entries, $maxEntries = null, $checkModEssOnly = false)
    {
        $logEntries = array();

        if (!empty($entries))
        {
            $visitor = XenForo_Visitor::getInstance();
            $threadModel = $this->_getThreadModel();
            $userModel = $this->_getUserModel();
            $userProfileModel = $this->_getUserProfileModel();
            $threadIds = array();
            $forumIds = array();
            $userIds = array();

            $canViewResources = $visitor->hasPermission('resource', 'view');
            $canViewWarnings = XenForo_Application::get('options')->modess_log_warnings['modlog'] && $visitor->hasPermission('general', 'viewWarning') ? true : false;

            // collect ids so we can then remove those the user cannot see.
            foreach ($entries AS $entryId => &$entry)
            {
                $type = $entry['content_type'];

                if ($checkModEssOnly && $type != 'modess')
                {
                    continue;
                }

                if ($type == 'modess')
                {
                    if (!$canViewWarnings)
                    {
                        unset($entries[$entryId]);
                        continue;
                    }

                    $type = $entry['discussion_content_type'];
                }

                // collect ids
                if ($type == 'thread' || $type == 'post')
                {
                    // collect forum or thread ids
                    $elements = json_decode($entry['action_params'], true);
                    if (!empty($elements) && isset($elements['node_id']))
                    {
                        $nodeId = $elements['node_id'];
                        $forumIds[$nodeId] = $nodeId;
                        $entry['node_id'] = $nodeId;
                    }
                    else
                    {
                        $threadIds[$entry['discussion_content_id']] = $entry['discussion_content_id'];
                    }
                }
                else if ($type == 'profile_post' || $type == 'user') // collect all user ids
                {
                    $userIds[$entry['discussion_content_id']] = $entry['discussion_content_id'];
                }
                else if ($type == 'resource' || $type == 'resource_update')
                {
                    if (!$canViewResources)
                        unset($entries[$entryId]);
                }
            }

            // only keep those we cannot see
            foreach ($forumIds AS $forumId)
            {
                // unset those we can see
                if ($threadModel->canViewThreadModeratorLog(array('node_id' => $forumId), array()))
                    unset($forumIds[$forumId]);
            }

            // only keep those we cannot see
            $nodeIds = array();
            foreach ($threadIds AS $threadId)
            {
                $thread = $threadModel->getThreadById($threadId);
                if (!empty($thread))
                {
                    if (array_key_exists($thread['node_id'], $nodeIds))
                    {
                        if ($nodeIds[$thread['node_id']] == 'unset')
                            unset($threadIds[$threadId]); // we already know we can see this forum
                    }
                    else
                    {
                        // unset those we can see
                        if ($threadModel->canViewThreadModeratorLog(array('node_id' => $thread['node_id']), array()))
                        {
                            unset($threadIds[$threadId]);
                            $nodeIds[$thread['node_id']] = 'unset';
                        }
                        else
                        {
                            $nodeIds[$thread['node_id']] = 'keep';
                        }
                    }
                }
            }

            // only keep those we cannot see
            foreach ($userIds AS $userId)
            {
                $errorPhraseKey = '';
                $fetchOptions = array('join' => XenForo_Model_User::FETCH_USER_FULL | XenForo_Model_User::FETCH_USER_PERMISSIONS);
                $profileUser = $userModel->getUserById($userId, $fetchOptions);

                // unset those we can see
                if (!empty($profileUser) && !empty($profileUser['user_id']) && $userProfileModel->canViewFullUserProfile($profileUser, $errorPhraseKey, $visitor->toArray()))
                {
                    unset($userIds[$userId]);
                }
            }

            // add those we can see into an ordered array
            $numEntries = 0;
            foreach ($entries AS $logId => $log)
            {
                $type = $log['content_type'];
                if ($type == 'modess')
                {
                    $type = $log['discussion_content_type'];
                }

                if ($type == 'post' || $type == 'thread')
                {
                    if (isset($log['node_id']))
                    {
                        if (!array_key_exists($log['node_id'], $forumIds))
                        {
                            $logEntries[$logId] = $log;
                            $numEntries = $numEntries + 1;
                        }
                    }
                    else
                    {
                        if (!array_key_exists($log['discussion_content_id'], $threadIds))
                        {
                            $logEntries[$logId] = $log;
                            $numEntries = $numEntries + 1;
                        }
                    }

                    if (!isset($logEntries[$logId]) && $log['content_type'] == 'modess')
                    {
                        // show regardless (as warning info is present in member profile) but remove thread and forum info
                        if ($log['action'] == 'warn' || $log['action'] == 'warn_update' || $log['action'] == 'warn_delete')
                        {
                            $log['content_id'] = 0;
                            $logEntries[$logId] = $log;
                            $numEntries = $numEntries + 1;
                        }
                    }
                }
                else if ($type == 'profile_post' || $type == 'user')
                {
                    if (!array_key_exists($log['discussion_content_id'], $userIds))
                    {
                        $logEntries[$logId] = $log;
                        $numEntries = $numEntries + 1;
                    }
                }
                else if ($type == 'resource' || $type == 'resource_update')
                {
                    $logEntries[$logId] = $log;
                    $numEntries = $numEntries + 1;
                }

                // limit to maximum items
                if (isset($maxEntries) && $numEntries > $maxEntries)
                    break;
            }
        }

        return $logEntries;
    }

    public function getModLogEntriesByIds(array $ids, $doCheck = true)
    {
        if (!$ids)
        {
            return array();
        }

        $entries = $this->fetchAllKeyed('
				SELECT moderator_log.*, user.username, user.avatar_date, user.avatar_width, user.avatar_height, user.gravatar
				FROM xf_moderator_log AS moderator_log
				INNER JOIN xf_user AS user ON (user.user_id = moderator_log.user_id)
				WHERE moderator_log_id IN (' . $this->_getDb()->quote($ids) . ')
				ORDER BY moderator_log.log_date DESC,  moderator_log.moderator_log_id DESC
		', 'moderator_log_id');

        if ($doCheck && !empty($entries))
        {
            // since warnings are shown regardless (as they appear on user profiles), we must remove thread and forum info if they cannot be viewed
            $entries = $this->_weedOutUnViewableEntries($entries, null, true); // 'null' for no max limit, 'true' for only modess type
        }

        return $entries;
    }

    /**
     * Log the action.
     *
     * @return boolean
     */
    public function logModeratorActionLocal(
        $contentType, array $content, $action, array $actionParams = array(),
        $parentContent = null, array $logUser = null
    )
    {
        // save the forum info to display in the session mod log
        if ($contentType == 'unc') // we do not count username changes
        {
            return parent::logModeratorActionLocal($contentType, $content, $action, $actionParams, $parentContent, $logUser);
        }
        else if ($contentType == 'thread')
        {
            switch ($action)
            {
                case 'posts_checked': // AVForums_ModCheckpoint add-on are not counted
                case 'bulk_posts_checked':
                {
                    return parent::logModeratorActionLocal($contentType, $content, $action, $actionParams, $parentContent, $logUser);
                }
                case 'move': // need to fetch the new node id and title since $content is the old thread info
                {
                    $thread = $this->_getThreadModel()->getThreadById($content['thread_id'], array('join' => XenForo_Model_Thread::FETCH_FORUM));
                    if (!empty($thread))
                    {
                        $actionParams['node_id'] = $thread['node_id'];
                        $actionParams['node_title'] = $thread['node_title'];
                    }
                    break;
                }
                case 'title': // need to fetch the new thread title since $content is the old thread info
                {
                    $thread = $this->_getThreadModel()->getThreadById($content['thread_id']);
                    if (!empty($thread))
                    {
                        $actionParams['new_title'] = $thread['title'];
                    }
                }
                case 'merge_target': // in the event the threads are kept as redirects then save their titles
                {
                    if (isset($actionParams['ids']))
                    {
                        $threadModel = $this->_getThreadModel();
                        $threadTitles = array();
                        $ids = explode(',', $actionParams['ids']);

                        foreach ($ids AS $id)
                        {
                            $id = intval(trim($id));
                            $thread = $threadModel->getThreadById($id);
                            if (!empty($thread))
                                $threadTitles[$id] = $thread['title'];
                        }

                        if (!empty($threadTitles))
                            $actionParams['thread_titles'] = $threadTitles;
                    }
                }
                case 'prefix':
                {
                    $thread = $this->_getThreadModel()->getThreadById($content['thread_id']);
                    if (!empty($thread) && $thread['prefix_id'])
                    {
                        $phrase = new XenForo_Phrase('thread_prefix_' . $thread['prefix_id']);
                        $phrase = $phrase->render();
                    }
                    else
                        $phrase = '-';

                    $actionParams['new_prefix'] = $phrase;
                }
                default: // all of above except 'move' fall through down to this one
                    {
                    $nodeId = $content['node_id'];
                    $actionParams['node_id'] = $nodeId;
                    $forum = $this->_getForumModel()->getForumById($nodeId);
                    if (!empty($forum))
                        $actionParams['node_title'] = $forum['title'];
                    }
            }
        }
        else if ($contentType == 'post')
        {
            $threadId = $content['thread_id'];
            $thread = $this->_getThreadModel()->getThreadById($threadId, array('join' => XenForo_Model_Thread::FETCH_FORUM));
            if (!empty($thread))
            {
                $actionParams['node_id'] = $thread['node_id'];
                $actionParams['node_title'] = $thread['node_title'];
            }
        }
        else if ($contentType == 'resource' || $contentType == 'resource_update')
        {
            if ($contentType == 'resource_update' || $action != 'delete_hard')
            {
                $category = $this->_getResourceModel()->getResourceById($content['resource_id'], array('join' => XenResource_Model_Resource::FETCH_CATEGORY));
            }
            else
            {
                $category = $this->_getResourceCategoryModel()->getCategoryById($content['resource_category_id']);
            }

            if (!empty($category))
            {
                $actionParams['category_id'] = $category['resource_category_id'];
                $actionParams['category_title'] = $category['category_title'];

                if ($contentType == 'resource_update')
                {
                    $actionParams['resource_id'] = $category['resource_id'];
                    $actionParams['resource_title'] = $category['title'];
                }
            }
        }

        $logId = parent::logModeratorActionLocal($contentType, $content, $action, $actionParams, $parentContent, $logUser);

        // add to the moderator log total count in the session
        if ($logId) // moderator_log_id or false
        {
            $this->_getModEssModel()->updateSessionModLogCounts($logId, $action, $contentType, $content, $logUser);
        }

        return $logId;
    }

    /**
     * Gets the latest log entry date and user id.
     *
     * @return integer
     */
    public function getLatestModLog()
    {
        return $this->_getDb()->fetchRow('
			SELECT moderator_log_id AS lastLogId, user_id AS userId
			FROM xf_moderator_log
			ORDER BY log_date DESC, moderator_log_id DESC
			LIMIT 1
		');
    }

    /**
     * Add content links to Session Mod Log entries.
     *
     * @param array $entries
     *
     * @return array
     */
    public function addLinksToModLogEntries(array $entries)
    {
        foreach ($entries AS &$entry)
        {
            $action = $entry['action'];
            $contentType = $entry['content_type'];

            if ($contentType == 'thread')
            {
                switch ($action)
                {
                    case 'reply_ban':
                    case 'reply_ban_delete':
                    {
                        $elements = json_decode($entry['action_params'], true);

                        if (!empty($elements) && isset($elements['name']))
                        {
                            $entry['action'] = 'modess_' . $action;

                            if (!isset($elements['url']))
                            {
                                $elements['url'] = XenForo_Link::buildPublicLink('members', '', array('username' => $elements['name']));
                                $entry['action_params'] = json_encode($elements);
                            }
                        }

                        $entry['linkToThread'] = $this->_buildLinkToThread($entry['discussion_content_id'], $entry['content_title']);
                        $entry['linkToForum'] = $this->_buildLinkToForum($entry);
                        break;
                    }
                    case 'post_copy_target_existing':
                    case 'post_move_target_existing':
                    {
                        $elements = json_decode($entry['action_params'], true);

                        if (!empty($elements) && isset($elements['ids']))
                        {
                            // add the links to the posts
                            $postIds = explode(',', $elements['ids']);
                            $postLinks = '';
                            foreach ($postIds AS $id)
                            {
                                $id = intval(trim($id));
                                if ($postLinks)
                                    $postLinks .= ', ';

                                $postLinks .= '<a href="' . XenForo_Link::buildPublicLink('posts', array('post_id' => $id)) . '">' . $id . '</a>';
                            }

                            if ($postLinks)
                                $entry['linkToContent'] = '(' . $postLinks . ')';

                            // use a phrase that does not display the unlinked post ids
                            if ($action == 'post_move_target_existing')
                                $entry['action'] = 'modess_post_move_target_existing';
                            else
                                $entry['action'] = 'modess_post_copy_target_existing';
                        }

                        $entry['linkToThread'] = $this->_buildLinkToThread($entry['discussion_content_id'], $entry['content_title']);
                        $entry['linkToForum'] = $this->_buildLinkToForum($entry);
                        break;
                    }
                    case 'post_copy_target':
                    case 'post_move_target':
                    {
                        $elements = json_decode($entry['action_params'], true);

                        if (!empty($elements) && isset($elements['ids']))
                        {
                            // add the links to the posts
                            $postIds = explode(',', $elements['ids']);
                            $postLinks = '';
                            foreach ($postIds AS $id)
                            {
                                $id = intval(trim($id));
                                if ($postLinks)
                                    $postLinks .= ', ';

                                $postLinks .= '<a href="' . XenForo_Link::buildPublicLink('posts', array('post_id' => $id)) . '">' . $id . '</a>';
                            }

                            if ($postLinks)
                                $entry['linkToContent'] = '(' . $postLinks . ')';

                            // use a phrase that does not display the unlinked post ids
                            if ($action == 'post_move_target')
                                $entry['action'] = 'modess_post_move_target';
                            else
                                $entry['action'] = 'modess_post_copy_target';
                        }

                        $entry['linkToThread'] = $this->_buildLinkToThread($entry['discussion_content_id'], $entry['content_title']);
                        $entry['linkToForum'] = $this->_buildLinkToForum($entry);
                        break;
                    }
                    case 'move':
                    {
                        $elements = json_decode($entry['action_params'], true);

                        // provide a link to the forum
                        if (!empty($elements) && isset($elements['from']))
                        {
                            $forumName = $elements['from'];

                            if (isset($this->_nodeIdsCache[$forumName]))
                            {
                                $nodeId = $this->_nodeIdsCache[$forumName];
                            }
                            else
                            {
                                $nodeId = $this->_getNodeModel()->getNodeIdFromTitle($forumName);
                                $this->_nodeIdsCache[$forumName] = $nodeId;
                            }

                            if ($nodeId)
                            {
                                $entry['action'] = 'modess_move_thread';
                                $entry['linkToContent'] = '<a href="' . XenForo_Link::buildPublicLink('forums', array('node_id' => $nodeId)) . '">' . $forumName . '</a>';
                            }
                        }

                        $entry['linkToThread'] = $this->_buildLinkToThread($entry['discussion_content_id'], $entry['content_title']);
                        $entry['linkToForum'] = $this->_buildLinkToForum($entry);
                        break;
                    }
                    case 'merge_target':
                    {
                        $elements = json_decode($entry['action_params'], true);

                        if (!empty($elements) && isset($elements['ids']) && isset($elements['thread_titles']))
                        {
                            // add a tooltip of thread titles - only available when the threads were kept as redirects
                            $threadTitles = $elements['thread_titles'];
                            $threadIds = explode(',', $elements['ids']);
                            $threadLinks = '';
                            foreach ($threadIds AS $id)
                            {
                                $id = intval(trim($id));
                                if ($threadLinks)
                                    $threadLinks .= ', ';

                                $threadLinks .= '<span style="cursor:pointer;" class="Tooltip" title="' . $threadTitles[$id] . '">' . $id . '</span>';
                            }

                            if ($threadLinks)
                                $entry['linkToContent'] = '(' . $threadLinks . ')';

                            // use a phrase that does not display the unlinked thread ids
                            $entry['action'] = 'modess_thread_merge_target';
                        }

                        $entry['linkToThread'] = $this->_buildLinkToThread($entry['discussion_content_id'], $entry['content_title']);
                        $entry['linkToForum'] = $this->_buildLinkToForum($entry);
                        break;
                    }
                    case 'post_copy_source':
                    case 'post_move_source':
                    {
                        $elements = json_decode($entry['action_params'], true);

                        // shorten the thread title
                        if (!empty($elements) && isset($elements['title']))
                        {
                            $elements['title'] = XenForo_Helper_String::wholeWordTrim($elements['title'], 65);
                            $entry['action_params'] = json_encode($elements);
                        }

                        $entry['linkToThread'] = $this->_buildLinkToThread($entry['discussion_content_id'], $entry['content_title']);
                        $entry['linkToForum'] = $this->_buildLinkToForum($entry);
                        break;
                    }
                    case 'prefix':
                    {
                        $elements = json_decode($entry['action_params'], true);
                        if (!empty($elements) && isset($elements['new_prefix']))
                        {
                            $entry['linkToContent'] = " to '" . $elements['new_prefix'] . "'";
                        }

                        $entry['linkToThread'] = $this->_buildLinkToThread($entry['discussion_content_id'], $entry['content_title']);
                        $entry['linkToForum'] = $this->_buildLinkToForum($entry);
                        break;
                    }
                    case 'title':
                    {
                        // use the new thread title
                        $elements = json_decode($entry['action_params'], true);
                        if (!empty($elements) && isset($elements['new_title']))
                        {
                            $title = $elements['new_title'];
                        }
                        else
                            $title = $entry['content_title'];

                        $entry['linkToThread'] = $this->_buildLinkToThread($entry['discussion_content_id'], $title);
                        $entry['linkToForum'] = $this->_buildLinkToForum($entry);
                        break;
                    }
                    /*
                    case 'delete_soft':
                    case 'delete_hard':
                    case 'undelete':
                    case 'stick':
                    case 'unstick':
                    case 'approve':
                    case 'unapprove':
                    case 'lock':
                    case 'unlock':
                    case 'poll':
                    case 'poll_add':
                    case 'poll_edit':
                    case 'threadban':
                    case 'threadban_edit':
                    case 'threadbanlift':
                    case 'threadbanlift_cron':
                    */
                    default:
                        {
                        $entry['linkToThread'] = $this->_buildLinkToThread($entry['discussion_content_id'], $entry['content_title']);
                        $entry['linkToForum'] = $this->_buildLinkToForum($entry);
                        break;
                        }
                }
            }
            else if ($contentType == 'post')
            {
                switch ($action)
                {
                    case 'delete_soft':
                    case 'undelete':
                    case 'approve':
                    case 'unapprove':
                    case 'edit':
                    {
                        $entry['linkToContent'] = '(<a href="' . $entry['content_url'] . '">' . $entry['content_id'] . '</a>)';
                        $entry['linkToThread'] = $this->_buildLinkToThread($entry['discussion_content_id'], $entry['content_title']);
                        $entry['linkToForum'] = $this->_buildLinkToForum($entry);
                        break;
                    }
                    case 'merge_target':
                    {
                        // use a phrase that does not display the unlinked post ids
                        $entry['action'] = 'modess_post_merge_target';
                        $elements = json_decode($entry['action_params'], true);

                        if (!empty($elements) && isset($elements['ids']))
                        {
                            // add the links to the posts
                            $postIds = explode(',', $elements['ids']);
                            $postLinks = '';
                            foreach ($postIds AS $id)
                            {
                                $id = intval(trim($id));
                                if ($postLinks)
                                    $postLinks .= ', ';

                                $postLinks .= '<a href="' . XenForo_Link::buildPublicLink('posts', '', array('post_id' => $entry['content_id'])) . '">' . $id . '</a>';
                            }

                            if ($postLinks)
                                $entry['linkToContent'] = '(' . $postLinks . ')';
                        }

                        $entry['linkToThread'] = $this->_buildLinkToThread($entry['discussion_content_id'], $entry['content_title']);
                        $entry['linkToForum'] = $this->_buildLinkToForum($entry);
                        break;
                    }
                    case 'delete_hard':
                    {
                        $entry['linkToContent'] = '(' . $entry['content_id'] . ')';
                        $entry['linkToThread'] = $this->_buildLinkToThread($entry['discussion_content_id'], $entry['content_title']);
                        $entry['linkToForum'] = $this->_buildLinkToForum($entry);
                        break;
                    }
                }
            }
            else if ($contentType == 'profile_post')
            {
                if ($action == 'delete_hard')
                    $entry['linkToContent'] = '(' . $entry['content_id'] . ')';
                else
                    $entry['linkToContent'] = '(<a href="' . $entry['content_url'] . '">' . $entry['content_id'] . '</a>)';

                $entry['linkToProfile'] = $this->_buildLinkToProfile($entry['discussion_content_id'], $entry['content_title']);
            }
            else if ($contentType == 'resource')
            {
                if ($action == 'delete_hard')
                {
                    $entry['linkToResource'] = $entry['content_title'];
                }
                else
                {
                    $entry['linkToResource'] = '<a href="' . $entry['content_url'] . '">' . $entry['content_title'] . '</a>';
                }

                $entry['linkToResourceCategory'] = $this->_buildLinkToResourceCategory($entry);
            }
            else if ($contentType == 'resource_update')
            {
                $entry['linkToResource'] = $this->_buildLinkToResource($entry);

                if (!$entry['linkToResource'] || $action == 'delete_hard')
                {
                    $entry['linkToResourceUpdate'] = $entry['content_title'];
                }
                else
                {
                    $entry['linkToResourceUpdate'] = '<a href="' . $entry['content_url'] . '">' . $entry['content_title'] . '</a>';
                }

                $entry['linkToResourceCategory'] = $this->_buildLinkToResourceCategory($entry);
            }
            else if ($contentType == 'modess')
            {
                if ($action == 'warn' || $action == 'warn_update' || $action == 'warn_delete')
                {
                    $elements = json_decode($entry['action_params'], true);
                    $this->_buildLinkToThread($entry['content_id'], $entry['content_title']);

                    if (!empty($elements))
                    {
                        switch ($entry['discussion_content_type'])
                        {
                            case 'post':
                            {
                                if ($entry['content_id']) // can be viewed - was decided in _weedOutUnViewableEntries()
                                {
                                    if (isset($elements['post_id']))
                                        $entry['linkToThread'] = '<a href="' . XenForo_Link::buildPublicLink('posts', array('post_id' => $elements['post_id'])) . '">' . $entry['content_title'] . '</a>';
                                    else
                                        $entry['linkToThread'] = $this->_buildLinkToThread($entry['discussion_content_id'], $entry['content_title']);

                                    $entry['linkToForum'] = $this->_buildLinkToForum($entry);
                                }
                                else
                                {
                                    $entry['linkToThread'] = new XenForo_Phrase('n_a');
                                    $entry['linkToForum'] = new XenForo_Phrase('n_a');
                                }
                                break;
                            }
                            case 'user':
                            {
                                $entry['linkToProfile'] = $this->_buildLinkToProfile($entry['discussion_content_id'], $entry['content_title']);
                                break;
                            }
                            case 'profile_post':
                            {
                                $entry['linkToProfile'] = '<a href="' . XenForo_Link::buildPublicLink('profile-posts', array('profile_post_id' => $entry['content_id'])) . '">' . $entry['content_title'] . '</a>';
                                break;
                            }
                            case 'resource_update':
                            {
                                if (isset($elements['is_actual_update']) && $elements['is_actual_update'])
                                    $entry['linkToResourceUpdate'] = '<a href="' . $entry['content_url'] . '">' . $entry['content_title'] . '</a>';

                                $entry['linkToResource'] = $this->_buildLinkToResource($entry);
                                $entry['linkToResourceCategory'] = $this->_buildLinkToResourceCategory($entry);
                                break;
                            }
                        }
                    }
                }
            }
        }

        return $entries;
    }

    protected function _buildLinkToThread($threadId, $title)
    {
        return '<a href="' . XenForo_Link::buildPublicLink('threads', array('thread_id' => $threadId)) . '">' . $title . '</a>';
    }

    protected function _buildLinkToForum(array $entry)
    {
        $elements = json_decode($entry['action_params'], true);
        if (!empty($elements) && isset($elements['node_id']))
        {
            $nodeId = $elements['node_id'];

            if (isset($this->_forumLinksCache[$nodeId]))
            {
                return $this->_forumLinksCache[$nodeId];
            }
            else
            {
                if (isset($elements['node_title']))
                {
                    $title = $elements['node_title'];
                }
                else
                {
                    $forum = $this->_getForumModel()->getForumById($nodeId);
                    $title = $forum['title'];
                }

                $link = '<a href="' . XenForo_Link::buildPublicLink('forums', array('node_id' => $nodeId)) . '">' . $title . '</a>';
                $this->_forumLinksCache[$nodeId] = $link;
                return $link;
            }
        }
        else
        {
            if (isset($entry['discussion_content_id']))
                return $this->_buildLinkToForumByThreadId($entry);
            else
                return '';
        }
    }

    // for entries made before this add-on
    protected function _buildLinkToForumByThreadId(array $entry)
    {
        $thread = $this->_getThreadModel()->getThreadById($entry['discussion_content_id'], array('join' => XenForo_Model_Thread::FETCH_FORUM));
        if (!empty($thread))
        {
            $nodeId = $thread['node_id'];
            $title = $thread['node_title'];
            $link = '<a href="' . XenForo_Link::buildPublicLink('forums', array('node_id' => $nodeId)) . '">' . $title . '</a>';

            // update the entry with forum info so we do not have to do this again next time
            $dw = XenForo_DataWriter::create('XenForo_DataWriter_ModeratorLog', XenForo_DataWriter::ERROR_SILENT);
            $dw->setExistingData($entry['moderator_log_id']);

            $actionParams = json_decode($dw->get('action_params'), true);
            $actionParams['node_id'] = $nodeId;
            $actionParams['node_title'] = $title;
            $actionParams = json_encode($actionParams);

            $dw->set('action_params', $actionParams);
            $dw->save();

            return $link;
        }
        return '';
    }

    protected function _buildLinkToResource(array $entry)
    {
        $elements = json_decode($entry['action_params'], true);
        if (!empty($elements) && isset($elements['resource_id']))
        {
            $resourceId = $elements['resource_id'];

            if (isset($this->_resourceLinksCache[$resourceId]))
            {
                return $this->_resourceLinksCache[$resourceId];
            }
            else
            {
                if (isset($elements['resource_title']))
                {
                    $title = $elements['resource_title'];
                }
                else
                {
                    $resource = $this->_getResourceModel()->getResourceById($resourceId);
                    $title = $resource['title'];
                }

                $link = '<a href="' . XenForo_Link::buildPublicLink('resources', array('resource_id' => $resourceId)) . '">' . $title . '</a>';
                $this->_resourceLinksCache[$resourceId] = $link;
                return $link;
            }
        }
        else
        {
            if (isset($entry['discussion_content_id']))
                return $this->_buildLinkToResourceByResourceId($entry);
            else
                return '';
        }
    }

    // for entries made before this add-on
    protected function _buildLinkToResourceByResourceId(array $entry)
    {
        $resource = $this->_getResourceModel()->getResourceById($entry['discussion_content_id']);
        if (!empty($resource))
        {
            $resourceId = $resource['resource_id'];
            $title = $resource['title'];
            $link = '<a href="' . XenForo_Link::buildPublicLink('resources', $resource) . '">' . $title . '</a>';
            $this->_resourceLinksCache[$resourceId] = $link;

            // update the entry with forum info so we do not have to do this again next time
            $dw = XenForo_DataWriter::create('XenForo_DataWriter_ModeratorLog', XenForo_DataWriter::ERROR_SILENT);
            $dw->setExistingData($entry['moderator_log_id']);

            $actionParams = json_decode($dw->get('action_params'), true);
            $actionParams['resource_id'] = $resourceId;
            $actionParams['resource_title'] = $title;
            $actionParams = json_encode($actionParams);

            $dw->set('action_params', $actionParams);
            $dw->save();

            return $link;
        }
        return '';
    }

    protected function _buildLinkToResourceCategory(array $entry)
    {
        $elements = json_decode($entry['action_params'], true);
        if (!empty($elements) && isset($elements['category_id']))
        {
            $categoryId = $elements['category_id'];

            if (isset($this->_resourceCategoryLinksCache[$categoryId]))
            {
                return $this->_resourceCategoryLinksCache[$categoryId];
            }
            else
            {
                if (isset($elements['category_title']))
                {
                    $title = $elements['category_title'];
                }
                else
                {
                    $category = $this->_getResourceCategoryModel()->getCategoryById($categoryId);
                    $title = $category['title'];
                }

                $link = '<a href="' . XenForo_Link::buildPublicLink('resources/categories', array('resource_category_id' => $categoryId)) . '">' . $title . '</a>';
                $this->_resourceCategoryLinksCache[$categoryId] = $link;
                return $link;
            }
        }
        else
        {
            if (isset($entry['discussion_content_id']))
                return $this->_buildLinkToResourceCategoryByResourceId($entry);
            else
                return '';
        }
    }

    // for entries made before this add-on
    protected function _buildLinkToResourceCategoryByResourceId(array $entry)
    {
        $resource = $this->_getResourceModel()->getResourceById($entry['discussion_content_id'], array('join' => XenResource_Model_Resource::FETCH_CATEGORY));
        if (!empty($resource))
        {
            $categoryId = $resource['resource_category_id'];
            $title = $resource['category_title'];
            $link = '<a href="' . XenForo_Link::buildPublicLink('resources/categories', array('resource_category_id' => $categoryId)) . '">' . $title . '</a>';
            $this->_resourceCategoryLinksCache[$categoryId] = $link;

            // update the entry with forum info so we do not have to do this again next time
            $dw = XenForo_DataWriter::create('XenForo_DataWriter_ModeratorLog', XenForo_DataWriter::ERROR_SILENT);
            $dw->setExistingData($entry['moderator_log_id']);

            $actionParams = json_decode($dw->get('action_params'), true);
            $actionParams['category_id'] = $categoryId;
            $actionParams['category_title'] = $title;
            $actionParams = json_encode($actionParams);

            $dw->set('action_params', $actionParams);
            $dw->save();

            return $link;
        }
        return '';
    }

    protected function _buildLinkToProfile($userId, $userName)
    {
        if (isset($this->_profileLinksCache[$userId]))
        {
            return $this->_profileLinksCache[$userId];
        }
        else
        {
            $link = '<a href="' . XenForo_Link::buildPublicLink('members', array('user_id' => $userId)) . '">' . $userName . '</a>';
            $this->_profileLinksCache[$userId] = $link;
            return $link;
        }
    }

    public function pruneModeratorLogEntries($pruneDate = null)
    {
        parent::pruneModeratorLogEntries($pruneDate);

        if ($pruneDate === null)
        {
            $logLength = XenForo_Application::get('options')->moderatorLogLength;
            if (!$logLength)
                return;

            $pruneDate = XenForo_Application::$time - 86400 * $logLength;
        }

        $modLogCounts = $this->_getDataRegistryModel()->get('modLogCounts');

        if (!empty($modLogCounts) && is_array($modLogCounts))
        {
            $modEssModel = $this->_getModEssModel();

            // recalculate counts for those mods who's lastViewDate is smaller than prune date
            foreach ($modLogCounts AS $userId => $logCounts)
            {
                if ($logCounts['lastViewDate'] <= $pruneDate)
                    $modEssModel->rebuildSessionModLogCountsCache($userId, true); // true for use registry to fetch
            }
        }
    }

    public function getModeratorLogForDiscussionContent($discussionType, $discussionId)
    {
        if (XenForo_Application::get('options')->modess_log_warnings['modlog'])
        {
            $visitor = XenForo_Visitor::getInstance();

            if ($visitor->hasPermission('general', 'viewWarning'))
            {
                $entries = parent::getModeratorLogForDiscussionContent($discussionType, $discussionId);

                $modess = $this->fetchAllKeyed('
					SELECT moderator_log.*, user.username
					FROM xf_moderator_log AS moderator_log
					INNER JOIN xf_user AS user ON (user.user_id = moderator_log.user_id)
					WHERE moderator_log.content_type = ?
						AND moderator_log.discussion_content_type = ?
						AND moderator_log.discussion_content_id = ?
					ORDER BY moderator_log.log_date DESC
				', 'moderator_log_id', array('modess', 'post', $discussionId));

                $entries = $entries + $modess;
                krsort($entries);
                return $entries;
            }
        }

        return parent::getModeratorLogForDiscussionContent($discussionType, $discussionId);
    }
    /*
        public function getModeratorLogEntries($userId = 0, array $fetchOptions = array())
        {
            $entries = parent::getModeratorLogEntries($userId, $fetchOptions);

            $visitor = XenForo_Visitor::getInstance();
            if (!$visitor->hasPermission('general', 'viewWarning'))
            {
                $db = $this->_getDb();
                $limitOptions = $this->prepareLimitFetchOptions($fetchOptions);

                return $this->fetchAllKeyed($this->limitQueryResults(
                    '
                        SELECT moderator_log.*, user.username
                        FROM xf_moderator_log AS moderator_log
                        INNER JOIN xf_user AS user ON (user.user_id = moderator_log.user_id)
                        WHERE ' . ($userId ? 'moderator_log.user_id = ' . $db->quote($userId) : '1=1') . ' AND (content_type != ' . $db->quote('modess') . ' OR (action != ' . $db->quote('warn') . ' AND action != ' . $db->quote('warn_update') . ' AND action != ' . $db->quote('warn_delete') . '))
                        ORDER BY moderator_log.log_date DESC
                    ', $limitOptions['limit'], $limitOptions['offset']
                ), 'moderator_log_id');
            }

            return $entries;
        }

        public function countModeratorLogEntries($userId = 0)
        {
            $visitor = XenForo_Visitor::getInstance();
            if (!$visitor->hasPermission('general', 'viewWarning'))
            {
                $db = $this->_getDb();

                return $db->fetchOne('
                    SELECT COUNT(*)
                    FROM xf_moderator_log
                    WHERE ' . ($userId ? 'user_id = ' . $db->quote($userId) : '1=1') . ' AND (content_type != ' . $db->quote('modess') . ' OR (action != ' . $db->quote('warn') . ' AND action != ' . $db->quote('warn_update') . ' AND action != ' . $db->quote('warn_delete') . '))'
                );
            }
            else
            {
                return parent::countModeratorLogEntries($userId);
            }
        }
    */

    /**
     * @return XenForo_Model_User
     */
    protected function _getUserModel()
    {
        return $this->getModelFromCache('XenForo_Model_User');
    }

    /**
     * @return XenForo_Model_UserProfile
     */
    protected function _getUserProfileModel()
    {
        return $this->getModelFromCache('XenForo_Model_UserProfile');
    }

    /**
     * @return XenForo_Model_Thread
     */
    protected function _getThreadModel()
    {
        return $this->getModelFromCache('XenForo_Model_Thread');
    }

    /**
     * @return XenForo_Model_Forum
     */
    protected function _getForumModel()
    {
        return $this->getModelFromCache('XenForo_Model_Forum');
    }

    /**
     * @return XenForo_Model_Node
     */
    protected function _getNodeModel()
    {
        return $this->getModelFromCache('XenForo_Model_Node');
    }

    /**
     * @return XenForo_Model_PermissionCache
     */
    protected function _getPermissionCacheModel()
    {
        return $this->getModelFromCache('XenForo_Model_PermissionCache');
    }

    /**
     * @return ModEss_Model_ModEss
     */
    protected function _getModEssModel()
    {
        return $this->getModelFromCache('ModEss_Model_ModEss');
    }

    /**
     * @return XenResource_Model_Resource
     */
    protected function _getResourceModel()
    {
        return $this->getModelFromCache('XenResource_Model_Resource');
    }

    /**
     * @return XenResource_Model_Category
     */
    protected function _getResourceCategoryModel()
    {
        return $this->getModelFromCache('XenResource_Model_Category');
    }
}