<?php

/**
 * Controller for handling actions on threads.
 *
 * @package ModEss
 */
class ModEss_ControllerPublic_Thread extends XFCP_ModEss_ControllerPublic_Thread
{

    /**
     * check moderator have permission to participate in thread
     * @param type $action
     * @throws type
     */
    protected function _preDispatch($action)
    {
        parent::_preDispatch($action);
        $ThreadId = $this->_input->filterSingle('thread_id', XenForo_Input::UINT);
        $options = XenForo_Application::getOptions();
        $visitor = XenForo_Visitor::getInstance();

        if (!$visitor->hasPermission('general', 'canModeratorDiscussion'))
        {
            $entry = $this->getModelFromCache('ModEss_Model_ModeratorDiscussion')->getCreatedDiscussion($ThreadId, $options->mod_discuss_node);
            if($entry)
            {
                throw $this->getNoPermissionResponseException();
            }
        }
    }

    /**
     * Displays a thread.
     *
     * @return XenForo_ControllerResponse_Abstract
     */
    public function actionIndex()
    {
        $response = parent::actionIndex();

        if (!empty($response->params) && isset($response->params['thread']))
        {
            $thread = $response->params['thread'];
            if (isset($thread['node_id'])) // some people seem to have a conflict with another add-on which returns a thread with no node_id
            {
                $visitor = XenForo_Visitor::getInstance();

                // show deleted posts link in thread tools
                $response->params['canViewDeletedPosts'] = $visitor->hasNodePermission($thread['node_id'], 'viewDeleted');
            }

            if (!empty($response->params['thread']) && !empty($response->params['forum']))
            {
                $response->params['modessCanViewContributors'] = $this->_getThreadModel()->modessCanViewPosters($response->params['thread'], $response->params['forum'], $errorPhraseKey);
                $response->params['modessCanViewModLogs'] = $this->_getThreadModel()->modessCanViewModLogs($response->params['thread'], $response->params['forum'], $errorPhraseKey);
            }
        }

        return $response;
    }

    //start thread discussion for moderator    
    public function actionStartThreadDiscussion()
    {

        $visitor = XenForo_Visitor::getInstance();
        if (!$visitor->hasPermission('general', 'canModeratorDiscussion'))
        {
            return $this->responseNoPermission();
        }
        $ThreadId = $this->_input->filterSingle('thread_id', XenForo_Input::UINT);


        $options = XenForo_Application::getOptions();
        $forum = $this->getModelFromCache('XenForo_Model_Forum')->getForumById($options->mod_discuss_node);

        if ($options->mod_discuss_node > 0 and $forum['node_type_id'] == "Forum")
        {

            $entry = $this->getModelFromCache('ModEss_Model_ModeratorDiscussion')->getDiscussionByThread($ThreadId, $options->mod_discuss_node);
            $thread = $this->getModelFromCache('XenForo_Model_Thread')->getThreadById($ThreadId);

            if (!$entry)
            {


                $writer = XenForo_DataWriter::create('XenForo_DataWriter_Discussion_Thread');
                $writer->bulkSet(array(
                    'user_id' => $visitor['user_id'],
                    'username' => $visitor['username'],
                    'title' => $thread['title'],
                    'prefix_id' => 0,
                    'discussion_type' => 'modess_action_discussion',
                    'node_id' => $options->mod_discuss_node
                ));
                $postWriter = $writer->getFirstMessageDw();
                $postWriter->set('message', "thread-id_" . $ThreadId);
                $writer->save();
                $discussion_thread = $writer->get('thread_id');




                $thread = $writer->getMergedData();

                $dw = XenForo_DataWriter::create('ModEss_DataWriter_ModeratorDiscussion');
                $dw->set('thread_id', $ThreadId);
                $dw->set('forum_id', $options->mod_discuss_node);
                $dw->set('user_id', $visitor['user_id']);
                $dw->set('discussion_thread', $discussion_thread);
                $dw->set('content_user_id', $thread['user_id']);
                $dw->set('content_username', $thread['username']);
                $dw->save();

                return $this->responseRedirect(
                                XenForo_ControllerResponse_Redirect::SUCCESS, XenForo_Link::buildPublicLink('threads', $thread), new XenForo_Phrase('modess_the_discussion_for_this_action_has_been_opened')
                );
            }
            else
            {

                return $this->responseRedirect(
                                XenForo_ControllerResponse_Redirect::SUCCESS, XenForo_Link::buildPublicLink('threads', array('thread_id' => $entry['discussion_thread']))
                );
            }
        }
        else
        {
            return $this->responseError(new XenForo_Phrase('moderation_node_type'));
        }
    }

    public function actionModeratorLog()
    {
        $threadId = $this->_input->filterSingle('thread_id', XenForo_Input::UINT);

        $ftpHelper = $this->getHelper('ForumThreadPost');
        list($thread, $forum) = $ftpHelper->assertThreadValidAndViewable($threadId);

        if (!$this->_getThreadModel()->modessCanViewModLogs($thread, $forum, $errorPhraseKey))
        {
            throw $this->getErrorOrNoPermissionResponseException($errorPhraseKey);
        }

        $logModel = $this->_getLogModel();
        $modessModel = $this->getModelFromCache('ModEss_Model_ModEss');

        $id = $this->_input->filterSingle('id', XenForo_Input::UINT);
        if ($id)
        {
            $entry = $logModel->getModeratorLogById($id);
            if (!$entry)
            {
                return $this->responseError(new XenForo_Phrase('requested_log_entry_not_found'), 404);
            }

            $viewParams = array(
                'entry' => $logModel->prepareModeratorLogEntry($entry)
            );
            return $this->responseView('XenForo_ViewAdmin_Log_ModeratorView', 'modess_log_moderator_view', $viewParams);
        }
        else
        {
            $threadId = $this->_input->filterSingle('thread_id', XenForo_Input::UINT);

            $page = $this->_input->filterSingle('page', XenForo_Input::UINT);
            $perPage = 20;

            $pageParams = array();

            $entries = $modessModel->getThreadModeratorLogEntries($threadId, array('page' => $page, 'perPage' => $perPage));

            $viewParams = array(
                'entries' => $logModel->prepareModeratorLogEntries($entries),
                'total' => $modessModel->countThreadModeratorLogEntries($threadId),
                'page' => $page,
                'perPage' => $perPage,
                'pageParams' => $pageParams,
                'thread' => $thread,
                'logUsers' => $logModel->getUsersWithModeratorLogs(),
                'nodeBreadCrumbs' => $ftpHelper->getNodeBreadCrumbs($forum)
            );

            return $this->responseView('XenForo_ViewAdmin_Log_Moderator', 'modess_log_moderator', $viewParams);
        }
    }

    public function actionContributors()
    {
        $threadId = $this->_input->filterSingle('thread_id', XenForo_Input::UINT);

        $ftpHelper = $this->getHelper('ForumThreadPost');
        list($thread, $forum) = $ftpHelper->assertThreadValidAndViewable($threadId);

        if (!$this->_getThreadModel()->modessCanViewPosters($thread, $forum, $errorPhraseKey))
        {
            throw $this->getErrorOrNoPermissionResponseException($errorPhraseKey);
        }

        $posters = $this->_getThreadModel()->modessGetThreadPosters($thread['thread_id']);
        if (empty($posters))
        {
            throw $this->getErrorOrNoPermissionResponseException($errorPhraseKey);
        }

        $posterIds = array_keys($posters);

        /* @var XenForo_Model_User $userModel */
        $userModel = $this->getModelFromCache('XenForo_Model_User');

        $_posters = $userModel->getUsersByIds($posterIds, array(
            'join' => XenForo_Model_User::FETCH_USER_FULL
        ));

        if (!empty($posters))
        {
            foreach ($posters as &$poster)
            {
                if (isset($_posters[$poster['user_id']]))
                {
                    $poster = array_merge($_posters[$poster['user_id']], array('_post_count' => $poster['post_count']));
                }
                else
                {
                    unset($poster);
                }
            }
        }

        $viewParams = array(
            'posters' => $posters,
            'thread' => $thread,
            'node' => $forum,
            'nodeBreadCrumbs' => $ftpHelper->getNodeBreadCrumbs($forum)
        );

        return $this->responseView('ModEss_ViewPublic_Thread_Contributors', 'modess_thread_contributors', $viewParams);
    }

    /**
     * Updates an existing thread.
     *
     * @return XenForo_ControllerResponse_Abstract
     */
    public function actionSave()
    {
        if (!XenForo_Application::get('options')->modess_sticky_thread_order)
        {
            return parent::actionSave();
        }

        $threadId = $this->_input->filterSingle('thread_id', XenForo_Input::UINT);
        $ftpHelper = $this->getHelper('ForumThreadPost');
        list($thread, $forum) = $ftpHelper->assertThreadValidAndViewable($threadId);
        if (!$this->_getThreadModel()->canStickUnstickThread($thread, $forum))
        {
            return parent::actionSave();
        }

        $response = parent::actionSave();
        $doStickyOrder = false;

        if ($this->_input->filterSingle('_returnDiscussionListItem', XenForo_Input::UINT))
        {
            // ensures we passed the assertThreadValidAndViewable check inside the _returnDiscussionListItem section
            if (!empty($response->params) && isset($response->params['thread']))
            {
                $thread = $response->params['thread'];
                $threadId = $thread['thread_id'];
                $doStickyOrder = true;
            }
        }
        else
        {
            $doStickyOrder = true;
        }

        if ($doStickyOrder)
        {
            $sticky = $this->_input->filterSingle('sticky', XenForo_Input::UINT);
            $stickyOrder = $this->_input->filterSingle('sticky_order', XenForo_Input::UINT);
            $stickyOrderModel = $this->_getSTOModel();

            if ($sticky && $stickyOrder)
            {
                $stickyOrderModel->addOrEditStickyOrder($stickyOrder, $threadId);
            }
            else
            {
                $stickyOrderModel->deleteStickyOrder($threadId);
            }
        }

        return $response;
    }

    /**
     * @return ModEss_Model_STO
     */
    protected function _getSTOModel()
    {
        return $this->getModelFromCache('ModEss_Model_STO');
    }

    /**
     * @return XenForo_Model_Node
     */
    protected function _getNodeModel()
    {
        return $this->getModelFromCache('XenForo_Model_Node');
    }

    /**
     * @return XenForo_Model_Log
     */
    protected function _getLogModel()
    {
        return $this->getModelFromCache('XenForo_Model_Log');
    }

}
