<?php

class XenPushover_Model_Pushover extends XenForo_Model
{
	protected $_priorityId = array(
		'low' => -1,
		'normal' => 0,
		'high' => 1,
		'emergency' => 2
	);

	public function getPriorityFromId($id)
	{
		$priorities = array_flip($this->_priorityId);

		return $priorities[$id];
	}

	public function getAlertType($contentType, $action)
	{
		return $this->_getDb()->fetchRow('
			SELECT *
			FROM xenpushover_alert_types
			WHERE content_type = ?
				AND action = ?
		', array($contentType, $action));
	}

	public function getAllAlertTypes()
	{
		return $this->_getDb()->fetchAll('
			SELECT *
			FROM xenpushover_alert_types
			ORDER BY content_type
		');
	}

	public function getAllAdminAlertTypes()
	{
		return $this->fetchAllKeyed('
			SELECT *
			FROM xenpushover_alert_admin
		', 'alert_type');
	}

	public function getAdminAlertType($alertType)
	{
		return $this->_getDb()->fetchRow('
			SELECT *
			FROM xenpushover_alert_admin
			WHERE alert_type = ?
		', $alertType);
	}

	public function prepareAdminAlertTypes(array $adminAlerts)
	{
		foreach ($adminAlerts AS &$adminAlert)
		{
			$adminAlert['priority_id'] = $this->_priorityId[$adminAlert['priority']];
		}

		return $adminAlerts;
	}

	public function getAdminAlertFromReceipt($receipt)
	{
		return $this->_getDb()->fetchRow('
			SELECT *
			FROM xenpushover_alert_admin
			WHERE emergency_receipt = ?
		', $receipt);
	}

	public function prepareForToggling($alertTypes = array())
	{
		$prepared = array();
		foreach ($alertTypes AS $alertType)
		{
			$prepared[$alertType['content_type'] . '_' . $alertType['action']] = $alertType;
		}

		return $prepared;
	}

	public function rebuildAlertTypes()
	{
		$alertTypes = $this->_getAlertTypes();

		$db = $this->_getDb();
		foreach ($alertTypes AS $alertType)
		{
			$db->query('
				INSERT IGNORE INTO xenpushover_alert_types
					(content_type, action, enabled)
				VALUES
					(?, ?, ?)
			', array($alertType['content_type'], $alertType['action'], $alertType['enabled']));
		}

		$alertTemplates = $db->fetchPairs('
			SELECT title, title
			FROM xf_template
			WHERE title LIKE ' . XenForo_Db::quoteLike('alert_', 'r', $db) . '
		');

		$exisitngAlertTypes = $this->getAllAlertTypes();
		foreach ($exisitngAlertTypes AS $existing)
		{
			if (!isset($alertTemplates['alert_' . $existing['content_type'] . '_' . $existing['action']]))
			{
				$db->delete('xenpushover_alert_types',
					'content_type = ' . $db->quote($existing['content_type']) . ' AND action = ' . $db->quote($existing['action'])
				);
			}
		}

		return true;
	}

	protected function _getAlertTypes()
	{
		/** @var $alertModel XenForo_Model_Alert */
		$alertModel = $this->getModelFromCache('XenForo_Model_Alert');

		$alertHandlers = $alertModel->getContentTypesWithField('alert_handler_class');

		$alertTypes = array();
		$regExpClause = '';
		foreach ($alertHandlers AS $contentType => $class)
		{
			if ($contentType == 'conversation')
			{
				continue;
			}

			$templatePrefix = 'alert_' . $contentType . '_';

			$alertTypes[$contentType]['templatePrefix'] = $templatePrefix;
			$regExpClause .= $templatePrefix . '|';
		}

		$db = $this->_getDb();

		$templates = $db->fetchPairs('
			SELECT title, title
			FROM xf_template
			WHERE title REGEXP ' . $db->quote(rtrim($regExpClause, '|')) . '
		');

		$data = array();

		foreach ($alertTypes AS $contentType => &$alertType)
		{
			foreach ($templates AS $template)
			{
				if (strstr($template, $alertType['templatePrefix']))
				{
					$action = str_replace($alertType['templatePrefix'], '', $template);
					$data[] = array(
						'content_type' => $contentType,
						'action' => $action,
						'enabled' => 1
					);
				}
			}
		}

		return $data;
	}

	/**
	 * Gets the friendly name phrase name.
	 *
	 * @param string $contentType
	 * @param string $action
	 *
	 * @return string
	 */
	public function getFriendlyNamePhraseName($contentType, $action = '')
	{
		$phraseName = 'xenpushover_friendly_';
		if ($action)
		{
			$phraseName .= $contentType . '_' . $action;
		}
		else
		{
			$phraseName .= 'ct_' . $contentType;
		}

		return $phraseName;
	}

	public function getScrapeUsers()
	{
		return $this->fetchAllKeyed('
			SELECT auth.*, user.*,
				permission_combination.cache_value AS global_permission_cache
			FROM xf_user_external_auth AS auth
			LEFT JOIN xf_user AS user ON
				(user.user_id = auth.user_id)
			LEFT JOIN xf_permission_combination AS permission_combination ON
				(permission_combination.permission_combination_id = user.permission_combination_id)
			WHERE user.user_state = \'valid\'
				AND user.is_banned = 0
				AND auth.provider = \'xenpushoverScrape\'
		', 'user_id');
	}

	public function getNotificationSounds()
	{
		// TODO: This is available programmatically from the Pushover API.

		return array(
			'' => '',
			'pushover' => 'Pushover (default)',
			'bike' => 'Bike',
			'bugle' => 'Bugle',
			'cashregister' => 'Cash Register',
			'classical' => 'Classical',
			'cosmic' => 'Cosmic',
			'falling' => 'Falling',
			'gamelan' => 'Gamelan',
			'incoming' => 'Incoming',
			'intermission' => 'Intermission',
			'magic' => 'Magic',
			'mechanical' => 'Mechanical',
			'pianobar' => 'Piano Bar',
			'siren' => 'Siren',
			'spacealam' => 'Space Alarm',
			'tugboat' => 'Tug Boat',
			'alien' => 'Alien Alarm (long)',
			'climb' => 'Climb (long)',
			'persistent' => 'Persistent (long)',
			'echo' => 'Echo (long)',
			'updown' => 'Up Down (long)',
			'none' => 'None (silent)'
		);
	}

	public function getPriorityLevels($includeEmergency = false)
	{
		$priorityLevels = array(
			-1 => new XenForo_Phrase('xenpushover_low_priority'),
			0 => new XenForo_Phrase('xenpushover_normal_priority'),
			1 => new XenForo_Phrase('xenpushover_high_priority')
		);

		if ($includeEmergency)
		{
			$priorityLevels[2] = new XenForo_Phrase('xenpushover_emergency_priority');
		}

		return $priorityLevels;
	}

	public function getAlertTypesForPushover()
	{
		$alertTypes = $this->_getDb()->fetchAll('
			SELECT *
			FROM xenpushover_alert_types
			ORDER BY content_type
		');

		$grouped = array();

		foreach ($alertTypes AS $alertType)
		{
			$grouped[$alertType['content_type']][$alertType['action']] = $alertType;
		}

		return $grouped;
	}

	public function getPushoverResponse($url, array $params)
	{
		$client = XenForo_Helper_Http::getClient($url);

		$client->setMethod(Zend_Http_Client::POST);
		$client->setConfig(array(
			'options' => array(
				'use_ssl' => false
			),
			'timeout' => 30
		));
		$client->setParameterPost($params);

		try
		{
			$request = $client->request();
		}
		catch (Exception $e)
		{
			return array('status' => 0);
		}

		$response = $request->getBody();
		$response = json_decode($response, true);

		return $response;
	}

	public function queue($alertId, $userId, $queueTime)
	{
		$delayQueue = 0;

		/** @var $userModel XenForo_Model_User */
		$userModel = $this->getModelFromCache('XenForo_Model_User');
		$user = $userModel->getUserById($userId, array('join' => XenForo_Model_User::FETCH_LAST_ACTIVITY));

		if (!empty($user['effective_last_activity']) && ($queueTime <= XenForo_Application::$time))
		{
			if ($user['effective_last_activity'] >= XenForo_Application::$time - (5 * 60))
			{
				$delayQueue += XenForo_Application::getOptions()->xenpushoverDelayTimer;
			}
		}

		$db = $this->_getDb();

		$exists = $db->fetchOne('
			SELECT queue_id
			FROM xenpushover_alert_queue
			WHERE alert_id = ?
				AND user_id = ?
		', array($alertId, $userId));

		if ($exists)
		{
			$db->update('xenpushover_alert_queue', array(
				'queue_date' => $queueTime + $delayQueue
			), 'alert_id = ' . $alertId . ' user_id = ' . $userId);
		}
		else
		{
			$db->insert('xenpushover_alert_queue', array(
				'alert_id' => $alertId,
				'user_id' => $userId,
				'queue_date' => $queueTime + $delayQueue
			));
		}

		XenForo_Application::defer('XenPushover_Deferred_Pushover', array(), 'XenPushoverQueue', false, XenForo_Application::$time);

		return;
	}

	public function getQueue()
	{
		return $this->fetchAllKeyed('
			SELECT *
			FROM xenpushover_alert_queue
			ORDER BY queue_date
		', 'queue_id');
	}

	public function getNextRunTime()
	{
		return $this->_getDb()->fetchOne('
			SELECT queue_date
			FROM xenpushover_alert_queue
			ORDER BY queue_date DESC
			LIMIT 0, 1
		');
	}

	public function pushAdminAlert($alertType)
	{
		$options = XenForo_Application::getOptions();

		$message = new XenForo_Phrase('xenpushover_admin_alert_message_' . $alertType['alert_type']);
		$title = new XenForo_Phrase('xenpushover_admin_alert_title_' . $alertType['alert_type'], array('name' => $options->boardTitle));
		$urlTitle = new XenForo_Phrase('xenpushover_admin_alert_url_title_' . $alertType['alert_type']);

		$params = array(
			'token' => $options->xenpushoverApiToken,
			'user' => $options->xenpushoverAdminKey
		);

		switch ($alertType['alert_type'])
		{
			case 'server_error':

				$response = $this->getPushoverResponse('https://api.pushover.net/1/users/validate.json', $params);
				if ($response['status'] === 1)
				{
					$params = $params + array(
						'sound' => !empty($alertType['sound']) ? $alertType['sound'] : '',
						'message' => $message->render(),
						'priority' => $this->_priorityId[$alertType['priority']],
						'title' => $title->render(),
						'url' => XenForo_Link::buildAdminLink('canonical:logs/server-error'),
						'url_title' => $urlTitle->render(),
						'timestamp' => XenForo_Application::$time
					);
					if ($params['priority'] === 2)
					{
						$params['retry'] = 600;
						$params['expire'] = 86400;
						$params['callback'] = XenForo_Link::buildPublicLink('full:xenpushover');
					}

					$response = $this->getPushoverResponse('https://api.pushover.net/1/messages.json', $params);
					if ($response['status'] === 1)
					{
						if ($params['priority'] === 2)
						{
							return $response['receipt'];
						}

						return true;
					}
				}

				break;

			case 'user_approval':

				$response = $this->getPushoverResponse('https://api.pushover.net/1/users/validate.json', $params);
				if ($response['status'] === 1)
				{
					$params = $params + array(
							'sound' => !empty($alertType['sound']) ? $alertType['sound'] : '',
							'message' => $message->render(),
							'priority' => $this->_priorityId[$alertType['priority']],
							'title' => $title->render(),
							'url' => XenForo_Link::buildAdminLink('canonical:users/moderated'),
							'url_title' => $urlTitle->render(),
							'timestamp' => XenForo_Application::$time
						);
					if ($params['priority'] === 2)
					{
						$params['retry'] = 3600;
						$params['expire'] = 86400;
						$params['callback'] = XenForo_Link::buildPublicLink('full:xenpushover');
					}

					$response = $this->getPushoverResponse('https://api.pushover.net/1/messages.json', $params);
					if ($response['status'] === 1)
					{
						if ($params['priority'] === 2)
						{
							return $response['receipt'];
						}

						return true;
					}
				}

				break;
		}

		return false;
	}

	public function push($alertId, $userId)
	{
		$user = $this->getModelFromCache('XenForo_Model_User')->getUserById($userId, array('join' => XenForo_Model_User::FETCH_USER_PERMISSIONS));
		$user['permissions'] = XenForo_Permission::unserializePermissions($user['global_permission_cache']);

		if (!XenForo_Permission::hasPermission($user['permissions'], 'general', 'associateXenPushover'))
		{
			return true; // No retry
		}

		$alerts = $this->getModelFromCache('XenForo_Model_Alert')->getAlertsForUser($user['user_id'], XenForo_Model_Alert::FETCH_MODE_RECENT, array(), $user);
		if (!isset($alerts['alerts'][$alertId]))
		{
			return true;
		}
		$alert = $alerts['alerts'][$alertId];
		if (!$this->userReceivesPushover($user, $alert))
		{
			return true;
		}

		/** @var $alertHandler XenForo_AlertHandler_Abstract */
		$alertHandler = $alerts['alertHandlers'][$alert['content_type']];

		/** @var XenForo_Model_UserExternal $externalAuthModel */
		$externalAuthModel = $this->getModelFromCache('XenForo_Model_UserExternal');

		$external = $externalAuthModel->getExternalAuthAssociationsForUser($user['user_id']);
		if (!empty($external['xenpushover']))
		{
			$extra = @unserialize($external['xenpushover']['extra_data']);
			if ($extra)
			{
				$options = XenForo_Application::get('options');
				$extra['token'] = $options->xenpushoverApiToken;

				$response = $this->getPushoverResponse('https://api.pushover.net/1/users/validate.json', $extra);
				if ($response['status'] === 1)
				{
					$canViewAlert = $alertHandler->canViewAlert($alert, $alert['content'], $alert['user']);
					if (!$canViewAlert)
					{
						return true;
					}

					$messageHtml = $this->_renderAlertTemplate($alert);

					$dom = new Zend_Dom_Query($messageHtml);
					$popupItemLink = $dom->query('.PopupItemLink');

					$url = XenForo_Link::buildPublicLink('full:account/alerts#alert' . $alertId);
					$urlPhrase = new XenForo_Phrase('xenpushover_view_this_alert');
					if ($popupItemLink->count())
					{
						$url = XenForo_Link::convertUriToAbsoluteUri($popupItemLink->current()->getAttribute('href'), true);
						$urlPhrase = new XenForo_Phrase('xenpushover_view_this_content');
					}

					$message = $this->_prepareMessage($messageHtml->render());

					$titlePhrase = new XenForo_Phrase('xenpushover_new_alert_at_x', array('board' => $options->boardTitle));

					$notification = array(
						'token' => $extra['token'],
						'user' => $extra['user'],
						'device' => $extra['device'],
						'sound' => !empty($extra['sound']) ? $extra['sound'] : '',
						'message' => $message,
						'priority' => $this->getUserAlertPriority($user, $alert),
						'title' => $titlePhrase->render(),
						'url' => $url,
						'url_title' => $urlPhrase->render(),
						'timestamp' => XenForo_Application::$time
					);

					$response = $this->getPushoverResponse('https://api.pushover.net/1/messages.json', $notification);
					if ($response['status'] === 1)
					{
						return true;
					}
					else
					{
						return false; // re-queue
					}
				}
				else
				{
					return true; // do not re-queue this is most likely a fatal error
				}
			}
			else
			{
				return true;
			}
		}

		return true;
	}

	public function pushScrapedAlert($alertHtml, $alertId, array $extraData)
	{
		$response = $this->getPushoverResponse('https://api.pushover.net/1/users/validate.json', $extraData);
		if ($response['status'] === 1)
		{
			$alertText = $this->_prepareMessage($alertHtml);

			$timeString  = utf8_substr($alertText, -10, 10);
			$alertText = str_replace($timeString, '', $alertText);
			$alertText = trim($alertText);

			$url = rtrim($extraData['url'], '/');
			if (strpos($url, 'index.php'))
			{
				$url .= '?';
			}
			else
			{
				$url .= '/';
			}
			$alertUrl = $url . 'account/alerts#' . $alertId;

			$urlPhrase = new XenForo_Phrase('xenpushover_view_this_alert');

			$dom = new Zend_Dom_Query($alertHtml);
			$popupItemLink = $dom->query('.PopupItemLink');

			if ($popupItemLink->count())
			{
				$alertUrl = $url . $popupItemLink->current()->getAttribute('href');
				$urlPhrase = new XenForo_Phrase('xenpushover_view_this_content');
			}

			$titlePhrase = new XenForo_Phrase('xenpushover_new_alert_at_x', array('board' => !empty($extraData['nick']) ? $extraData['nick'] : $extraData['url']));

			$notification = array(
				'token' => $extraData['token'],
				'user' => $extraData['user'],
				'device' => $extraData['device'],
				'sound' => !empty($extraData['sound']) ? $extraData['sound'] : '',
				'priority' => !empty($extraData['priority']) ? $extraData['priority'] : 0,
				'message' => $alertText,
				'title' => $titlePhrase->render(),
				'url' => $alertUrl,
				'url_title' => $urlPhrase->render(),
				'timestamp' => XenForo_Application::$time
			);

			$response = $this->getPushoverResponse('https://api.pushover.net/1/messages.json', $notification);
			if ($response['status'] === 1)
			{
				return true;
			}
		}
	}

	/**
	 * Renders a view...
	 *
	 * @param array $alert
	 * @return object XenForo_Template_Abstract
	 */
	protected function _renderAlertTemplate(array $alert)
	{
		$request = new Zend_Controller_Request_Http();
		$response = new Zend_Controller_Response_Http();

		$dependencies = new XenForo_Dependencies_Public();
		$dependencies->preRenderView();

		$renderer = $dependencies->getViewRenderer($response, 'Html', $request);
		$content = $renderer->renderView('', $alert, 'alert_' . $alert['content_type'] . '_' . $alert['action']);

		return $content;
	}

	protected function _prepareMessage($messageHtml)
	{
		$message = XenForo_Template_Helper_Core::helperStripHtml($messageHtml);
		$message = trim($message);
		$message = html_entity_decode($message, ENT_QUOTES, 'UTF-8');

		return $message;
	}

	public function prepareOptOuts(array $optOuts)
	{
		$db = $this->_getDb();

		ksort($optOuts);
		$optOuts = array_keys($optOuts);

		$userId = XenForo_Visitor::getUserId();

		$db->delete('xenpushover_alert_optout', 'user_id = ' . $userId);

		foreach ($optOuts AS $alert)
		{
			$db->insert('xenpushover_alert_optout', array(
				'user_id' => $userId,
				'alert' => $alert
			));
		}
	}

	public function preparePriorities(array $priorites)
	{
		$db = $this->_getDb();

		ksort($priorites);

		$userId = XenForo_Visitor::getUserId();

		$db->delete('xenpushover_alert_priority', 'user_id = ' . $userId);

		foreach ($priorites AS $alert => $priority)
		{
			$db->insert('xenpushover_alert_priority', array(
				'user_id' => $userId,
				'alert' => $alert,
				'priority' => $priority
			));
		}
	}

	public function getAlertOptOuts(array $user = null)
	{
		if ($user === null)
		{
			$user = XenForo_Visitor::getInstance();
		}

		if (!$user['user_id'])
		{
			return array();
		}
		else
		{
			$optOuts = $this->_getDb()->fetchCol('
				SELECT alert
				FROM xenpushover_alert_optout
				WHERE user_id = ?
			', $user['user_id']);
		}

		return array_fill_keys($optOuts, true);
	}

	public function getAlertPriorities(array $user = null)
	{
		if ($user === null)
		{
			$user = XenForo_Visitor::getInstance();
		}

		if (!$user['user_id'])
		{
			return array();
		}
		else
		{
			$priorities = $this->_getDb()->fetchPairs('
				SELECT alert, priority
				FROM xenpushover_alert_priority
				WHERE user_id = ?
			', $user['user_id']);
		}

		return $priorities;
	}

	public function mergeOptOutsAndPriorities($alertTypes, $optOuts, $priorities)
	{
		foreach ($alertTypes AS $contentType => &$alertType)
		{
			foreach ($alertType AS $action => &$item)
			{
				if (isset($optOuts[$contentType . '_' . $action]))
				{
					$item['opt_out'] = $optOuts[$contentType . '_' . $action];
				}

				if (isset($priorities[$contentType . '_' . $action]))
				{
					$item['priority'] = $priorities[$contentType . '_' . $action];
				}
			}
		}

		return $alertTypes;
	}

	public function userReceivesPushover($user, array $alert)
	{
		$alertTypes = $this->getAlertTypesForPushover();

		if (!empty($alertTypes[$alert['content_type']][$alert['action']]['enabled']) && !$alert['view_date'])
		{
			$optOuts = $this->getAlertOptOuts($user);

			return (empty($optOuts["{$alert['content_type']}_{$alert['action']}"]));
		}

		return false;
	}

	public function getUserAlertPriority($user, array $alert)
	{
		$alertPriorities = $this->getAlertPriorities($user);

		if (isset($alertPriorities["{$alert['content_type']}_{$alert['action']}"]))
		{
			return $alertPriorities["{$alert['content_type']}_{$alert['action']}"];
		}

		return 0;
	}
}