<?php
/**
* @package XenCentral Framework
* @author Skydevelop EU
* @copyright Drnoyan & Nalyan LDA, Portugal, EU
* @license http://www.dnf.pt/eula.html
* @link http://www.skydevelop.com
* @version 1.4.1
* @revision 125
*/
 
class XenCentral_Framework_Interface_Database
{
	protected $_db;
	
	public function __construct()
	{
		if (!class_exists('XenForo_Application'))
		{
			throw new Exception("Could not find XenForo backend");
		}
	}
	
	public function queryRead($q)
	{
		return $this->_getDb()->query($q);
	}

    public function resultCount($result)
    {
        $result->execute();
        $result->store_result();

        return $result->num_rows;
    }
	
	public function queryWrite($q)
	{
		return $this->_getDb()->query($q);
	}
	
	public function getFirst($q)
	{
		return $this->_getDb()->fetchRow($q);
	}
	
	public function fetchResult($resource)
	{
		return $resource->fetch(Zend_Db::FETCH_ASSOC);
	}
	
	public function escapeString($string)
	{
		return trim($this->_getDb()->quote($string), '\'');
	}
	
	public function lastInsertId()
	{
		return $this->_getDb()->lastInsertId();
	}

    public function getOne($q)
    {
        $result = $this->getFirst($q);
        if (is_array($result)) {
            return array_shift($result);
        }

        return $result;
    }

    public function insertOrUpdate($table, $array)
    {
        $arrayKeys = array_keys($array);
        $arrayValues = array_values($array);

        $arrayValues = array_map(array(
            $this,
            'escapeString'
        ), $arrayValues);

        $fieldList = '`' . implode('`, `', $arrayKeys) . '`';

        $valueList = '\'' . implode('\', \'', $arrayValues) . '\'';

        $updateClause = array();

        foreach ($arrayKeys as $key) {
            $updateClause[] = "`$key`=VALUES(`$key`)";
        }

        $updateClause = implode(", \n", $updateClause);

        return $this->queryWrite("
			INSERT INTO `$table` ($fieldList)
			VALUES ($valueList)
			ON DUPLICATE KEY UPDATE $updateClause
		");
    }

    public function update($table, $array, $condition)
    {
        $arrayKeys = array_keys($array);

        $arrayValues = array_values($array);

        $arrayValues = array_map(array(
            $this,
            'escapeString'
        ), $arrayValues);

        $updateClause = array();

        foreach ($arrayKeys as $index => $key) {
            $updateClause[] = "`$key`='" . $arrayValues[$index] . "'";
        }

        $updateClause = implode(", \n", $updateClause);

        return $this->queryWrite("
			UPDATE `$table`
			SET $updateClause
			WHERE $condition
		");
    }

    public function insert($table, $array)
    {
        $arrayKeys = array_keys($array);
        $arrayValues = array_values($array);

        $arrayValues = array_map(array(
            $this,
            'escapeString'
        ), $arrayValues);

        $fieldList = '`' . implode('`, `', $arrayKeys) . '`';

        $valueList = '\'' . implode('\', \'', $arrayValues) . '\'';

        return $this->queryWrite("
			INSERT INTO `$table` ($fieldList)
			VALUES ($valueList)
		");
    }

    public function upgradeTable($tableName, $tableDefinition)
    {
        // check if table exists
        $created = $this->getFirst("SHOW TABLES LIKE '$tableName'");

        if (!$created) {
            // just create the table
            $this->queryWrite($tableDefinition);
            return true;
        }

        // rename old table
        try {
            $created = $this->getFirst("SHOW TABLES LIKE '{$tableName}_backup'");
            if ($created) {
                // drop backup
                $this->queryWrite("DROP TABLE IF EXISTS {$tableName}_backup");
            }

            $this->queryWrite("RENAME TABLE  `$tableName` TO  `{$tableName}_backup` ;");
        } catch (Exception $ex) {
            // not have rename permission
            throw $ex;
        }

        // run table creation query
        try {
            $this->queryWrite($tableDefinition);
        } catch (Exception $ex) {
            // recover the table
            $this->queryWrite("RENAME TABLE  `{$tableName}_backup` TO  `{$tableName}` ;");
            throw $ex;
        }

        // check if table is successfully created
        $created = $this->getFirst("SHOW TABLES LIKE '$tableName'");

        if (!$created) {
            $this->queryWrite("RENAME TABLE  `{$tableName}_backup` TO  `{$tableName}` ;");

            return false;
        }

        $newCol = $oldCol = array();

        // try to move data
        $colResource = $this->queryRead("SHOW COLUMNS IN {$tableName}");
        while ($col = $this->fetchResult($colResource)) {
            $newCol[] = $col['Field'];
        }

        $colResource = $this->queryRead("SHOW COLUMNS IN {$tableName}_backup");
        while ($col = $this->fetchResult($colResource)) {
            $oldCol[] = $col['Field'];
        }

        $commonCols = array_intersect($newCol, $oldCol);

        try {
            $this->queryWrite("
				INSERT IGNORE INTO {$tableName} (`" . implode('`,`', $commonCols) . "`)
				SELECT `" . implode('`,`', $commonCols) . "` FROM {$tableName}_backup
			");
            $this->queryWrite("DROP TABLE IF EXISTS {$tableName}_backup");
        } catch (Exception $ex) {
            // failed to move the data, restore old table
            $this->queryWrite("DROP TABLE IF EXISTS {$tableName}");
            $this->queryWrite("RENAME TABLE  `{$tableName}_backup` TO  `{$tableName}` ;");
            throw $ex;
        }

        return true;
    }

    public function fetchAll($q, $key = '')
    {
        $rows = array();

        $resource = $this->queryRead($q);

        while ($row = $this->fetchResult($resource)) {
            if ($key) {
                if (!isset($row[$key])) {
                    throw new Exception("$key column not found in SQL output");
                }

                $rows[$row[$key]] = $row;
            } else {
                $rows[] = $row;
            }
        }

        return $rows;
    }

    public function delete($tableName, $condtion)
    {
        $this->queryWrite("
			DELETE FROM `$tableName`
			WHERE $condtion
		");

        return true;
    }
	
	/**
	* Helper method to get the database object.
	*
	* @return Zend_Db_Adapter_Abstract
	*/
	protected function _getDb()
	{
		if ($this->_db === null)
		{
			$this->_db = XenForo_Application::get('db');
		}

		return $this->_db; 
	}
}