<?php
/**
* @package XenCentral Framework
* @author Skydevelop EU
* @copyright Drnoyan & Nalyan LDA, Portugal, EU
* @license http://www.dnf.pt/eula.html
* @link http://www.skydevelop.com
* @version 1.4.1
* @revision 125
*/


/**
 * Controller for manager in the admin control panel.
 *
 * @package XenCentral
 */
abstract class XenCentral_Framework_ControllerAdmin_Manager extends XenForo_ControllerAdmin_Abstract
{
	/**
	 * @var XenCentral_Framework_Tools_ManagerInfo
	 */
	protected $_managerInfo;
	
	protected function _preDispatch($action)
	{
		parent::_preDispatch($action);
		
		$this->assertAdminPermission('option');
	}
	
	/**
	 * Lists all available items
	 *
	 * @return XenForo_ControllerResponse_Abstract
	 */
	public function actionIndex()
	{
		$itemModel = $this->_getItemModel();

		$viewParams = array(
			'items' => $itemModel->prepareItems($itemModel->getAllItems())
		);
		
		// allow child class to add params without overriding this function
		$this->additionalParams('index', $viewParams);

		return $this->responseView($this->getManagerInfo()->getManagerViewName().'_Index', $this->getManagerInfo()->getTemplatePrefix().'_index', $viewParams);
	}

	/**
	 * Gets the add/edit form response for a item.
	 *
	 * @param array $item
	 *
	 * @return XenForo_ControllerResponse_Abstract
	 */
	protected function _getItemAddEditResponse(array $item)
	{
		$itemModel = $this->_getItemModel();

		$viewParams = array(
			'item' => $item
		);
		
		$this->additionalParams('edit', $viewParams);

		return $this->responseView($this->getManagerInfo()->getManagerViewName().'_Edit', $this->getManagerInfo()->getTemplatePrefix().'_edit', $viewParams);
	}

	/**
	 * Displays a form to add a new item.
	 *
	 * @return XenForo_ControllerResponse_Abstract
	 */
	public function actionAdd()
	{
		return $this->_getItemAddEditResponse($this->_getItemModel()->getDefaultItem());
	}

	/**
	 * Displays a form to edit an existing item.
	 *
	 * @return XenForo_ControllerResponse_Abstract
	 */
	public function actionEdit()
	{
		$itemId = $this->_input->filterSingle($this->getManagerInfo()->getPrimaryKeyName(), XenForo_Input::UINT);
		$item = $this->_getItemOrError($itemId);

		return $this->_getItemAddEditResponse($item);
	}

	/**
	 * Inserts a new item or updates an existing one.
	 *
	 * @return XenForo_ControllerResponse_Abstract
	 */
	public function actionSave()
	{
		$this->_assertPostOnly();
		
		$itemModel = $this->_getItemModel();

		$itemId = $this->_input->filterSingle($this->getManagerInfo()->getPrimaryKeyName(), XenForo_Input::UINT);
		
		$dwData=$this->getDataArray();

		$dw = XenForo_DataWriter::create($this->getManagerInfo()->getDataWriterName(), XenForo_DataWriter::ERROR_ARRAY);
		
		$dw->setManagerInfo($this->getManagerInfo());
		
		if ($itemId)
		{
			$dw->setExistingData($itemId);
		}
		
		$dw->bulkSet($dwData);
		
		$dw->save();

		return $this->responseRedirect(
			XenForo_ControllerResponse_Redirect::SUCCESS,
			XenForo_Link::buildAdminLink($this->getManagerInfo()->getIndexLink())
		);
	}

	public function actionDelete()
	{
		if ($this->isConfirmedPost())
		{
			return $this->_deleteData(
				$this->getManagerInfo()->getDataWriterName(),
				$this->getManagerInfo()->getPrimaryKeyName(),
				XenForo_Link::buildAdminLink($this->getManagerInfo()->getIndexLink())
			);
		}
		else
		{
			$itemId = $this->_input->filterSingle($this->getManagerInfo()->getPrimaryKeyName(), XenForo_Input::UINT);
			$item = $this->_getItemOrError($itemId);

			$viewParams = array(
				'item' => $item
			);
			
			$this->additionalParams('delete', $viewParams);

			return $this->responseView($this->getManagerInfo()->getManagerViewName().'_Delete', $this->getManagerInfo()->getTemplatePrefix().'_delete', $viewParams);
		}
	}

	/**
	 * Gets the specified item or throws an exception.
	 *
	 * @param string $id
	 *
	 * @return array
	 */
	protected function _getItemOrError($id)
	{
		$itemModel = $this->_getItemModel();

		return $itemModel->prepareItem(
			$this->getRecordOrError(
				$id, $itemModel, 'getItemById',
				$this->getManagerInfo()->getNotFoundErrorPhrase()
			)
		);
	}

	/**
	 * @return XenForo_Model
	 */
	protected function _getItemModel()
	{
		$model=$this->getModelFromCache($this->getManagerInfo()->getModelClassName());
		
		$model->setManagerInfo($this->getManagerInfo());
		
		return $model;
	}
	
	/**
	 * Extend this function to add additional parameters for any action
	 */
	protected function additionalParams($action, &$params)
	{
		// add primary key id
		$params['_primaryId']=$this->getManagerInfo()->getPrimaryKeyName();
		
		// for each field defined in database add proper phrases to be used in template
		// $params['_phrases']=$this->getManagerInfo()->getDefaultPhrases();
	}
	
	final protected function _preDispatchFirst($action)
	{
		// setup manager info object
		$this->setManagerInfo();
	}
	
	protected function getManagerInfo()
	{
		return $this->_managerInfo;
	}
	
	protected abstract function setManagerInfo();
	
	protected function getDataArray()
	{
		$fields=$this->getManagerInfo()->getFields(true);
		
		$dwData=array();
		
		foreach ($fields AS $fieldId=>$fieldInfo)
		{
			$dwData[$fieldId]=$this->_input->filterSingle($fieldId, $fieldInfo['inputtype']);
		}
		
		return $dwData;
	}
}