<?php

/**
 * Model for resource managers.
 */
class Waindigo_ResMans_Model_ResourceManager extends XenForo_Model
{

    const SIMPLE_CACHE_KEY = 'waindigo_resMans_resourceManagers';

    /**
     * Gets resource managers that match the specified criteria.
     *
     * @param array $conditions List of conditions.
     * @param array $fetchOptions
     *
     * @return array [resource manager id] => info.
     */
    public function getResourceManagers(array $conditions = array(), array $fetchOptions = array())
    {
        $whereClause = $this->prepareResourceManagerConditions($conditions, $fetchOptions);

        $sqlClauses = $this->prepareResourceManagerFetchOptions($fetchOptions);
        $limitOptions = $this->prepareLimitFetchOptions($fetchOptions);

        return $this->fetchAllKeyed(
            $this->limitQueryResults(
                '
                SELECT resource_manager.*
                    ' . $sqlClauses['selectFields'] . '
                FROM xf_resource_manager_waindigo AS resource_manager
                ' . $sqlClauses['joinTables'] . '
                WHERE ' . $whereClause . '
                ' . $sqlClauses['orderClause'] . '
            ', $limitOptions['limit'], $limitOptions['offset']),
            'resource_manager_id');
    } /* END getResourceManagers */

    /**
     * Gets the resource manager that matches the specified criteria.
     *
     * @param array $conditions List of conditions.
     * @param array $fetchOptions Options that affect what is fetched.
     *
     * @return array false
     */
    public function getResourceManager(array $conditions = array(), array $fetchOptions = array())
    {
        $resourceManagers = $this->getResourceManagers($conditions, $fetchOptions);

        return reset($resourceManagers);
    } /* END getResourceManager */

    /**
     * Gets a resource manager by ID.
     *
     * @param integer $resourceManagerId
     * @param array $fetchOptions Options that affect what is fetched.
     *
     * @return array false
     */
    public function getResourceManagerById($resourceManagerId, array $fetchOptions = array())
    {
        $conditions = array(
            'resource_manager_id' => $resourceManagerId
        );

        return $this->getResourceManager($conditions, $fetchOptions);
    } /* END getResourceManagerById */

    public function getResourceManagerOptions()
    {
        $resourceManagers = $this->getResourceManagers();
        $options = array();
        foreach ($resourceManagers as $resourceManagerId => $resourceManager) {
            $options[$resourceManagerId] = array(
                'label' => $resourceManager['title'],
                'value' => $resourceManagerId
            );
        }
        return $options;
    } /* END getResourceManagerOptions */

    /**
     * Gets the total number of a resource manager that match the specified
     * criteria.
     *
     * @param array $conditions List of conditions.
     *
     * @return integer
     */
    public function countResourceManagers(array $conditions = array())
    {
        $fetchOptions = array();

        $whereClause = $this->prepareResourceManagerConditions($conditions, $fetchOptions);
        $joinOptions = $this->prepareResourceManagerFetchOptions($fetchOptions);

        $limitOptions = $this->prepareLimitFetchOptions($fetchOptions);

        return $this->_getDb()->fetchOne(
            '
            SELECT COUNT(*)
            FROM xf_resource_manager_waindigo AS resource_manager
            ' . $joinOptions['joinTables'] . '
            WHERE ' . $whereClause . '
        ');
    } /* END countResourceManagers */

    /**
     * Gets all resource managers titles.
     *
     * @return array [resource manager id] => title.
     */
    public static function getResourceManagerTitles()
    {
        $resourceManagers = XenForo_Model::create(__CLASS__)->getResourceManagers();
        $titles = array();
        foreach ($resourceManagers as $resourceManagerId => $resourceManager) {
            $titles[$resourceManagerId] = $resourceManager['title'];
        }
        return $titles;
    } /* END getResourceManagerTitles */

    /**
     * Gets the default resource manager record.
     *
     * @return array
     */
    public function getDefaultResourceManager()
    {
        return array(
            'resource_manager_id' => '', /* END 'resource_manager_id' */
            'default_tab' => 'last_update', /* END 'default_tab' */
        );
    } /* END getDefaultResourceManager */

    /**
     * Prepares a set of conditions to select resource managers against.
     *
     * @param array $conditions List of conditions.
     * @param array $fetchOptions The fetch options that have been provided. May
     * be edited if criteria requires.
     *
     * @return string Criteria as SQL for where clause
     */
    public function prepareResourceManagerConditions(array $conditions, array &$fetchOptions)
    {
        $db = $this->_getDb();
        $sqlConditions = array();

        if (isset($conditions['resource_manager_ids']) && !empty($conditions['resource_manager_ids'])) {
            $sqlConditions[] = 'resource_manager.resource_manager_id IN (' .
                 $db->quote($conditions['resource_manager_ids']) . ')';
        } else
            if (isset($conditions['resource_manager_id'])) {
                $sqlConditions[] = 'resource_manager.resource_manager_id = ' .
                     $db->quote($conditions['resource_manager_id']);
            }

        $this->_prepareResourceManagerConditions($conditions, $fetchOptions, $sqlConditions);

        return $this->getConditionsForClause($sqlConditions);
    } /* END prepareResourceManagerConditions */

    /**
     * Method designed to be overridden by child classes to add to set of
     * conditions.
     *
     * @param array $conditions List of conditions.
     * @param array $fetchOptions The fetch options that have been provided. May
     * be edited if criteria requires.
     * @param array $sqlConditions List of conditions as SQL snippets. May be
     * edited if criteria requires.
     */
    protected function _prepareResourceManagerConditions(array $conditions, array &$fetchOptions, array &$sqlConditions)
    {
    } /* END _prepareResourceManagerConditions */

    /**
     * Checks the 'join' key of the incoming array for the presence of the
     * FETCH_x bitfields in this class
     * and returns SQL snippets to join the specified tables if required.
     *
     * @param array $fetchOptions containing a 'join' integer key built from
     * this class's FETCH_x bitfields.
     *
     * @return string containing selectFields, joinTables, orderClause keys.
     * Example: selectFields = ', user.*, foo.title'; joinTables = ' INNER JOIN
     * foo ON (foo.id = other.id) '; orderClause = 'ORDER BY x.y'
     */
    public function prepareResourceManagerFetchOptions(array &$fetchOptions)
    {
        $selectFields = '';
        $joinTables = '';
        $orderBy = '';

        $this->_prepareResourceManagerFetchOptions($fetchOptions, $selectFields, $joinTables, $orderBy);

        return array(
            'selectFields' => $selectFields,
            'joinTables' => $joinTables,
            'orderClause' => ($orderBy ? "ORDER BY $orderBy" : '')
        );
    } /* END prepareResourceManagerFetchOptions */

    /**
     * Method designed to be overridden by child classes to add to SQL snippets.
     *
     * @param array $fetchOptions containing a 'join' integer key built from
     * this class's FETCH_x bitfields.
     * @param string $selectFields = ', user.*, foo.title'
     * @param string $joinTables = ' INNER JOIN foo ON (foo.id = other.id) '
     * @param string $orderBy = 'x.y ASC, x.z DESC'
     */
    protected function _prepareResourceManagerFetchOptions(array &$fetchOptions, &$selectFields, &$joinTables, &$orderBy)
    {
    } /* END _prepareResourceManagerFetchOptions */

    public function rebuildResourceManagerCache()
    {
        $resourceManagers = $this->getResourceManagers();

        XenForo_Application::setSimpleCacheData(self::SIMPLE_CACHE_KEY, $resourceManagers);
    } /* END rebuildResourceManagerCache */

    public function getCategoryAssociationsByResourceManager($resourceManagerId)
    {
        return $this->_getDb()->fetchCol(
            '
			SELECT resource_category_id
			FROM xf_resource_manager_category_waindigo
			WHERE resource_manager_id = ?
		', $resourceManagerId);
    } /* END getCategoryAssociationsByResourceManager */

    public function getCategoryAssociationsByResourceManagers(array $resourceManagerIds)
    {
        $db = $this->_getDb();

        $categories = $db->fetchCol(
            '
			SELECT resource_category_id
			FROM xf_resource_manager_category_waindigo
			WHERE resource_manager_id IN (' . $db->quote($resourceManagerIds) . ')
		');

        return array_unique($categories);
    } /* END getCategoryAssociationsByResourceManagers */

    public function verifyResourceManagerIsUsable(array $resourceManager, array $viewingUser = null)
    {
        $this->standardizeViewingUserReference($viewingUser);

        return $this->_verifyResourceManagerIsUsableInternal($resourceManager, $viewingUser);
    } /* END verifyResourceManagerIsUsable */

    protected function _verifyResourceManagerIsUsableInternal(array $resourceManager, array $viewingUser)
    {
        $userGroups = explode(',', $resourceManager['allowed_user_group_ids']);
        if (in_array(-1, $userGroups) || in_array($viewingUser['user_group_id'], $userGroups)) {
            return true; // available to all groups or the primary group
        }

        if ($viewingUser['secondary_group_ids']) {
            foreach (explode(',', $viewingUser['secondary_group_ids']) as $userGroupId) {
                if (in_array($userGroupId, $userGroups)) {
                    return true; // available to one secondary group
                }
            }
        }

        return false; // not available to any groups
    } /* END _verifyResourceManagerIsUsableInternal */
}