<?php

/**
 * Data writer for resource managers.
 */
class Waindigo_ResMans_DataWriter_ResourceManager extends XenForo_DataWriter
{

    const DATA_CATEGORY_IDS = 'categoryIds';

    /**
     * Title of the phrase that will be created when a call to set the
     * existing data fails (when the data doesn't exist).
     *
     * @var string
     */
    protected $_existingDataErrorPhrase = 'waindigo_requested_resource_manager_not_found_resmans';

    /**
     * Gets the fields that are defined for the table.
     * See parent for explanation.
     *
     * @return array
     */
    protected function _getFields()
    {
        return array(
            'xf_resource_manager_waindigo' => array(
                'resource_manager_id' => array(
                    'type' => self::TYPE_UINT,
                    'autoIncrement' => true
                ), /* END 'resource_manager_id' */
                'title' => array(
                    'type' => self::TYPE_STRING,
                    'required' => true
                ), /* END 'title' */
                'original_prefix' => array(
                    'type' => self::TYPE_STRING,
                    'required' => true
                ), /* END 'original_prefix' */
                'language_id' => array(
                    'type' => self::TYPE_UINT,
                    'default' => 0
                ), /* END 'language_id' */
                'style_id' => array(
                    'type' => self::TYPE_UINT,
                    'default' => 0
                ), /* END 'style_id' */
                'hide_tabs' => array(
                    'type' => self::TYPE_SERIALIZED
                ), /* END 'hide_tabs' */
                'default_tab' => array(
                    'type' => self::TYPE_STRING,
                    'required' => true
                ), /* END 'default_tab' */
                'allowed_user_group_ids' => array(
                    'type' => self::TYPE_UNKNOWN,
                    'default' => '',
                    'verification' => array(
                        '$this',
                        '_verifyAllowedUserGroupIds'
                    )
                )
            ), /* END 'xf_resource_manager_waindigo' */
        );
    } /* END _getFields */

    /**
     * Gets the actual existing data out of data that was passed in.
     * See parent for explanation.
     *
     * @param mixed
     *
     * @return array false
     */
    protected function _getExistingData($data)
    {
        if (!$resourceManagerId = $this->_getExistingPrimaryKey($data, 'resource_manager_id')) {
            return false;
        }

        $resourceManager = $this->_getResourceManagerModel()->getResourceManagerById($resourceManagerId);
        if (!$resourceManager) {
            return false;
        }

        return $this->getTablesDataFromArray($resourceManager);
    } /* END _getExistingData */

    /**
     * Gets SQL condition to update the existing record.
     *
     * @return string
     */
    protected function _getUpdateCondition($tableName)
    {
        return 'resource_manager_id = ' . $this->_db->quote($this->getExisting('resource_manager_id'));
    } /* END _getUpdateCondition */

    /**
     * Verifies the allowed user group IDs.
     *
     * @param array|string $userGroupIds Array or comma-delimited list
     *
     * @return boolean
     */
    protected function _verifyAllowedUserGroupIds(&$userGroupIds)
    {
        if (!is_array($userGroupIds))
        {
            $userGroupIds = preg_split('#,\s*#', $userGroupIds);
        }

        $userGroupIds = array_map('intval', $userGroupIds);
        $userGroupIds = array_unique($userGroupIds);
        sort($userGroupIds, SORT_NUMERIC);
        $userGroupIds = implode(',', $userGroupIds);

        return true;
    } /* END _verifyAllowedUserGroupIds */

    protected function _postSave()
    {
        $newCategoryIds = $this->getExtraData(self::DATA_CATEGORY_IDS);
        if (is_array($newCategoryIds)) {
            $this->_updateCategoryAssociations($newCategoryIds);
        }

        $this->_getResourceManagerModel()->rebuildResourceManagerCache();

        $this->_recompileTemplates();
    } /* END _postSave */

    protected function _updateCategoryAssociations(array $categoryIds)
    {
        $categoryIds = array_unique($categoryIds);

        $emptyNodeKey = array_search(0, $categoryIds);
        if ($emptyNodeKey !== false) {
            unset($categoryIds[$emptyNodeKey]);
        }

        $db = $this->_db;
        $resourceManagerId = $this->get('resource_manager_id');

        $db->delete('xf_resource_manager_category_waindigo', 'resource_manager_id = ' . $db->quote($resourceManagerId));

        foreach ($categoryIds as $categoryId) {
            $db->insert('xf_resource_manager_category_waindigo',
                array(
                    'resource_manager_id' => $resourceManagerId,
                    'resource_category_id' => $categoryId
                ));
        }

        return $categoryIds;
    } /* END _updateCategoryAssociations */

    protected function _postDelete()
    {
        $resourceManagerId = $this->get('resource_manager_id');

        $this->_db->delete('xf_resource_manager_category_waindigo',
            'resource_manager_id = ' . $this->_db->quote($resourceManagerId));

        $this->_getResourceManagerModel()->rebuildResourceManagerCache();

        $this->_recompileTemplates();
    } /* END _postDelete */

    protected function _recompileTemplates()
    {
        $templateModel = $this->getModelFromCache('XenForo_Model_Template');
        $templates = $templateModel->getTemplatesByTitles(array(
            'resources_tab_links'
        ));
        foreach ($templates as $template) {
            $templateModel->reparseTemplate($template, true);
        }
    } /* END _recompileTemplates */

    /**
     * Get the resource managers model.
     *
     * @return Waindigo_ResMans_Model_ResourceManager
     */
    protected function _getResourceManagerModel()
    {
        return $this->getModelFromCache('Waindigo_ResMans_Model_ResourceManager');
    } /* END _getResourceManagerModel */
}