<?php

/**
 * Admin controller for handling actions on resource managers.
 */
class Waindigo_ResMans_ControllerAdmin_ResourceManager extends XenForo_ControllerAdmin_Abstract
{

    /**
     * Shows a list of resource managers.
     *
     * @return XenForo_ControllerResponse_View
     */
    public function actionIndex()
    {
        $resourceManagerModel = $this->_getResourceManagerModel();
        $resourceManagers = $resourceManagerModel->getResourceManagers();
        $viewParams = array(
            'resourceManagers' => $resourceManagers
        );
        return $this->responseView('Waindigo_ResMans_ViewAdmin_ResourceManager_List',
            'waindigo_resource_manager_list_resmans', $viewParams);
    } /* END actionIndex */

    /**
     * Helper to get the resource manager add/edit form controller response.
     *
     * @param array $resourceManager
     *
     * @return XenForo_ControllerResponse_View
     */
    protected function _getResourceManagerAddEditResponse(array $resourceManager)
    {
        $languageModel = $this->_getLanguageModel();

        $userGroups = $this->_getUserGroupModel()->getAllUserGroups();

        if (!empty($resourceManager['resource_manager_id'])) {
            $selCategoryIds = $this->_getResourceManagerModel()->getCategoryAssociationsByResourceManager(
                $resourceManager['resource_manager_id']);

            $selUserGroupIds = explode(',', $resourceManager['allowed_user_group_ids']);
            if (in_array(-1, $selUserGroupIds)) {
                $allUserGroups = true;
                $selUserGroupIds = array_keys($userGroups);
            } else {
                $allUserGroups = false;
            }
        } else {
            $selCategoryIds = array();
            $allUserGroups = true;
            $selUserGroupIds = array_keys($userGroups);
        }

        if (!$selCategoryIds) {
            $selCategoryIds = array(
                0
            );
        }

        $hiddenTabs = array();
        if (!empty($resourceManager['hide_tabs'])) {
            $hiddenTabs = unserialize($resourceManager['hide_tabs']);
        }

        $viewParams = array(
            'resourceManager' => $resourceManager,

            'languageOptions' => $languageModel->getLanguagesForOptionsTag(),
            'styles' => $this->getModelFromCache('XenForo_Model_Style')->getAllStylesAsFlattenedTree(),

            'tabOrderChoiceOptions' => $this->_getResourceModel()->getTabOrderChoiceOptions($hiddenTabs, true),
            'defaultTabOrderOptions' => $this->_getResourceModel()->getTabOrderChoiceOptions(
                $resourceManager['default_tab']),

            'selCategoryIds' => $selCategoryIds,
            'allUserGroups' => $allUserGroups,
            'selUserGroupIds' => $selUserGroupIds,

            'categories' => $this->_getCategoryModel()->getAllCategories(),
            'userGroups' => $userGroups
        );

        return $this->responseView('Waindigo_ResMans_ViewAdmin_ResourceManager_Edit',
            'waindigo_resource_manager_edit_resmans', $viewParams);
    } /* END _getResourceManagerAddEditResponse */

    /**
     * Displays a form to add a new resource manager.
     *
     * @return XenForo_ControllerResponse_View
     */
    public function actionAdd()
    {
        $resourceManager = $this->_getResourceManagerModel()->getDefaultResourceManager();

        return $this->_getResourceManagerAddEditResponse($resourceManager);
    } /* END actionAdd */

    /**
     * Displays a form to edit an existing resource manager.
     *
     * @return XenForo_ControllerResponse_Abstract
     */
    public function actionEdit()
    {
        $resourceManagerId = $this->_input->filterSingle('resource_manager_id', XenForo_Input::STRING);

        if (!$resourceManagerId) {
            return $this->responseReroute('Waindigo_ResMans_ControllerAdmin_ResourceManager', 'add');
        }

        $resourceManager = $this->_getResourceManagerOrError($resourceManagerId);

        return $this->_getResourceManagerAddEditResponse($resourceManager);
    } /* END actionEdit */

    /**
     * Inserts a new resource manager or updates an existing one.
     *
     * @return XenForo_ControllerResponse_Abstract
     */
    public function actionSave()
    {
        $this->_assertPostOnly();

        $resourceManagerId = $this->_input->filterSingle('resource_manager_id', XenForo_Input::STRING);

        $input = $this->_input->filter(
            array(
                'title' => XenForo_Input::STRING,
                'original_prefix' => XenForo_Input::STRING,
                'language_id' => XenForo_Input::UINT,
                'style_id' => XenForo_Input::UINT,
                'show_tabs' => XenForo_Input::ARRAY_SIMPLE,
                'default_tab' => XenForo_Input::STRING,
                'usable_user_group_type' => XenForo_Input::STRING,
                'user_group_ids' => array(
                    XenForo_Input::UINT,
                    'array' => true
                )
            ));
        $categoryIds = $this->_input->filterSingle('resource_category_ids', XenForo_Input::UINT,
            array(
                'array' => true
            ));

        $input['hide_tabs'] = array();
        $tabOrderChoices = $this->_getResourceModel()->getTabOrderChoices();
        foreach ($tabOrderChoices as $key => $title) {
            if (!in_array($key, $input['show_tabs'])) {
                $input['hide_tabs'][] = $key;
            }
        }
        unset($input['show_tabs']);

        if ($input['usable_user_group_type'] == 'all') {
            $input['allowed_user_group_ids'] = array(
                -1
            );
        } else {
            $input['allowed_user_group_ids'] = $input['user_group_ids'];
        }
        unset($input['usable_user_group_type'], $input['user_group_ids']);

        $writer = XenForo_DataWriter::create('Waindigo_ResMans_DataWriter_ResourceManager');
        if ($resourceManagerId) {
            $writer->setExistingData($resourceManagerId);
        }
        $writer->setExtraData(Waindigo_ResMans_DataWriter_ResourceManager::DATA_CATEGORY_IDS, $categoryIds);
        $writer->bulkSet($input);
        $writer->save();

        if ($this->_input->filterSingle('reload', XenForo_Input::STRING)) {
            return $this->responseRedirect(XenForo_ControllerResponse_Redirect::RESOURCE_UPDATED,
                XenForo_Link::buildAdminLink('resource-managers/edit', $writer->getMergedData()));
        } else {
            return $this->responseRedirect(XenForo_ControllerResponse_Redirect::SUCCESS,
                XenForo_Link::buildAdminLink('resource-managers') .
                     $this->getLastHash($writer->get('resource_manager_id')));
        }
    } /* END actionSave */

    /**
     * Deletes a resource manager.
     *
     * @return XenForo_ControllerResponse_Abstract
     */
    public function actionDelete()
    {
        $resourceManagerId = $this->_input->filterSingle('resource_manager_id', XenForo_Input::STRING);
        $resourceManager = $this->_getResourceManagerOrError($resourceManagerId);

        $writer = XenForo_DataWriter::create('Waindigo_ResMans_DataWriter_ResourceManager');
        $writer->setExistingData($resourceManager);

        if ($this->isConfirmedPost()) { // delete resource manager
            $writer->delete();

            return $this->responseRedirect(XenForo_ControllerResponse_Redirect::SUCCESS,
                XenForo_Link::buildAdminLink('resource-managers'));
        } else { // show delete confirmation prompt
            $writer->preDelete();
            $errors = $writer->getErrors();
            if ($errors) {
                return $this->responseError($errors);
            }

            $viewParams = array(
                'resourceManager' => $resourceManager
            );

            return $this->responseView('Waindigo_ResMans_ViewAdmin_ResourceManager_Delete',
                'waindigo_resource_manager_delete_resmans', $viewParams);
        }
    } /* END actionDelete */

    /**
     * Gets a valid resource manager or throws an exception.
     *
     * @param string $resourceManagerId
     *
     * @return array
     */
    protected function _getResourceManagerOrError($resourceManagerId)
    {
        $resourceManager = $this->_getResourceManagerModel()->getResourceManagerById($resourceManagerId);
        if (!$resourceManager) {
            throw $this->responseException(
                $this->responseError(new XenForo_Phrase('waindigo_requested_resource_manager_not_found_resmans'), 404));
        }

        return $resourceManager;
    } /* END _getResourceManagerOrError */

    /**
     * Get the resource managers model.
     *
     * @return Waindigo_ResMans_Model_ResourceManager
     */
    protected function _getResourceManagerModel()
    {
        return $this->getModelFromCache('Waindigo_ResMans_Model_ResourceManager');
    } /* END _getResourceManagerModel */

    /**
     * Get the resources model.
     *
     * @return XenResource_Model_Resource
     */
    protected function _getResourceModel()
    {
        return $this->getModelFromCache('XenResource_Model_Resource');
    } /* END _getResourceModel */

    /**
     * Get the languages model.
     *
     * @return XenForo_Model_Language
     */
    protected function _getLanguageModel()
    {
        return $this->getModelFromCache('XenForo_Model_Language');
    } /* END _getLanguageModel */

    /**
     *
     * @return XenResource_Model_Category
     */
    protected function _getCategoryModel()
    {
        return $this->getModelFromCache('XenResource_Model_Category');
    } /* END _getCategoryModel */

    /**
     *
     * @return XenForo_Model_UserGroup
     */
    protected function _getUserGroupModel()
    {
        return $this->getModelFromCache('XenForo_Model_UserGroup');
    } /* END _getUserGroupModel */
}