<?php

/**
 * Model for posts.
 *
 * @package ModEss
 */
class ModEss_Model_Post extends XFCP_ModEss_Model_Post
{
	/**
	 * Prepares a post for display, generally within the context of a thread.
	 *
	 * @param array      $post   Post to prepare
	 * @param array      $thread Thread post is in
	 * @param array      $forum  Forum thread/post is in
	 * @param array|null $nodePermissions
	 * @param array|null $viewingUser
	 *
	 * @return array Prepared version of post
	 */
	public function preparePost(array $post, array $thread, array $forum, array $nodePermissions = null, array $viewingUser = null)
	{
		$post = parent::preparePost($post, $thread, $forum, $nodePermissions, $viewingUser);

		// whether or not to display the show post link
		$viewDeleted = $this->_getModEssModel()->viewDeleted($thread['node_id'], $nodePermissions, $viewingUser);
		$post['canViewDeleted'] = $viewDeleted;
		$post['canViewReason'] = ($viewDeleted ? true : XenForo_Application::get('options')->modess_delete_placeholder_view_reason);

		return $post;
	}

	/**
	 * Gets permission-based options that apply to post fetching functions.
	 *
	 * @param array      $thread Thread the posts will belong to
	 * @param array      $forum  Forum the thread belongs to
	 * @param array|null $nodePermissions
	 * @param array|null $viewingUser
	 *
	 * @return array Keys: deleted/moderated (both booleans)
	 */
	public function getPermissionBasedPostFetchOptions(array $thread, array $forum, array $nodePermissions = null, array $viewingUser = null)
	{
		$conditions = parent::getPermissionBasedPostFetchOptions($thread, $forum, $nodePermissions, $viewingUser);

		if (!$conditions['deleted'] && $this->_getModEssModel()
				->viewDeletedPostPlaceholders($thread['node_id'], $nodePermissions, $viewingUser)
		)
		{
			$conditions['deleted'] = true;
		}

		return $conditions;
	}

	public function preparePostJoinOptions(array $fetchOptions)
	{
		$parent = parent::preparePostJoinOptions($fetchOptions);

		#$parent['joinTables'] .= ' LEFT JOIN xf_user as modess_moderator on modess_moderator.user_id = post.last_edit_user_id';
		#$parent['selectFields'] .= ', IF(post.last_edit_user_id > 0, modess_moderator.username, post.username) AS last_edit_username';
		return $parent;
	}

	public function addInlineModOptionToPost(array &$post, array $thread, array $forum, array $nodePermissions = null, array $viewingUser = null)
	{
		$this->standardizeViewingUserReferenceForNode($thread['node_id'], $viewingUser, $nodePermissions);

		$postModOptions = parent::addInlineModOptionToPost($post, $thread, $forum, $nodePermissions,
			$viewingUser);

		$canInlineMod = ($viewingUser['user_id'] && (
				XenForo_Permission::hasContentPermission($nodePermissions, 'deleteAnyPost')
				|| XenForo_Permission::hasContentPermission($nodePermissions, 'undelete')
				|| XenForo_Permission::hasContentPermission($nodePermissions, 'approveUnapprove')
				|| XenForo_Permission::hasContentPermission($nodePermissions, 'manageAnyThread')
			));

		if ($canInlineMod)
		{
			if ($this->canConvertPostToConversation($post, $thread, $forum, $null, $nodePermissions,
				$viewingUser)
			)
			{
				$postModOptions['convertToConversation'] = true;
			}
		}

		$post['canInlineMod'] = (count($postModOptions) > 0);

		return $postModOptions;
	}

	public function canConvertPostToConversation(array $post, array $thread, array $forum, &$errorPhraseKey = '', array $nodePermissions = null, array $viewingUser = null)
	{
		$this->standardizeViewingUserReferenceForNode($thread['node_id'], $viewingUser, $nodePermissions);

		if ($this->isDeleted($post) || $this->isModerated($post))
		{
			$errorPhraseKey = 'modess_cant_convert_deleted_or_moderated_post_to_conversation';

			return false;
		}

		return XenForo_Permission::hasContentPermission($nodePermissions, 'modess_convertToConv');
	}

	/**
	 * @return ModEss_Model_ModEss
	 */
	protected function _getModEssModel()
	{
		return $this->getModelFromCache('ModEss_Model_ModEss');
	}
}

if (false)
{
	class XFCP_ModEss_Model_Post extends XenForo_Model_Post
	{
	}
}