<?php

/**
* Data writer for notifications
*
* @package GritterNotifications
*/
class GritterNotifications_DataWriter_Notification extends XenForo_DataWriter
{
	/**
	 * Title of the phrase that will be created when a call to set the
	 * existing data fails (when the data doesn't exist).
	 *
	 * @var string
	 */
	protected $_existingDataErrorPhrase = 'requested_notification_not_found';

	/**
	* Gets the fields that are defined for the table. See parent for explanation.
	*
	* @return array
	*/
	protected function _getFields()
	{
		return array(
			'xf_notification' => array(
				'notification_id'		=> array('type' => self::TYPE_UINT, 'autoIncrement' => true),
				'name'         			=> array('type' => self::TYPE_STRING, 'required' => true, 'maxLength' => 150, 'requiredError' => 'please_enter_valid_name'),
				'description' 	    	=> array('type' => self::TYPE_STRING),
				'active'        		=> array('type' => self::TYPE_BOOLEAN, 'default' => 1),
				'dismissible'			=> array('type' => self::TYPE_BOOLEAN, 'default' => 0),
				'sticky'				=> array('type' => self::TYPE_BOOLEAN, 'default' => 0),
				'style'					=> array('type' => self::TYPE_STRING, 'default' => 'gritter-dark',
					'allowedValues' 	=> array('gritter-dark', 'gritter-light', 'custom', 'easy')),
				'custom_style'			=> array('type' => self::TYPE_STRING),
				'custom_easy_bg'		=> array('type'	=> self::TYPE_STRING),
				'custom_easy_text'		=> array('type'	=> self::TYPE_STRING),
				'title'        			=> array('type' => self::TYPE_STRING, 'maxLength' => 150),
				'text'         			=> array('type' => self::TYPE_STRING, 'required' => true, 'maxLength' => 1500, 'requiredError' => 'please_enter_valid_text'),
				'showimage'  	      	=> array('type' => self::TYPE_STRING, 'default' => 'avatar',
					'allowedValues' 	=> array('avatar', '0', 'image')),
				'image'					=> array('type' => self::TYPE_STRING),
				'playsound'  	      	=> array('type' => self::TYPE_STRING, 'default' => '0'),
				'sound'					=> array('type' => self::TYPE_STRING),					
				'showtime'        		=> array('type' => self::TYPE_UINT, 'default' => 6000, 'max' => 3600000),
				'fadeintime'        	=> array('type' => self::TYPE_UINT, 'default' => 100, 'max' => 60000),
				'fadeouttime'        	=> array('type' => self::TYPE_UINT, 'default' => 100, 'max' => 60000),
				'tempclosetime'			=> array('type' => self::TYPE_UINT, 'default' => 0, 'max' => 100),
				'delay'       	 		=> array('type' => self::TYPE_UINT, 'default' => 0, 'max' => 3600000),
				'user_criteria' 		=> array('type' => self::TYPE_UNKNOWN, 'required' => true,
					'verification' 			=> array('$this', '_verifyCriteria')),
				'page_criteria' 		=> array('type' => self::TYPE_UNKNOWN, 'required' => true,
					'verification' 			=> array('$this', '_verifyCriteria')),
				'create_username'		=> array('type' => self::TYPE_STRING),
				'create_date'			=> array('type' => self::TYPE_INT),
				'edit_username'		=> array('type' => self::TYPE_STRING),
				'edit_date'			=> array('type' => self::TYPE_INT)					
			)
		);
	}

	/**
	* Gets the actual existing data out of data that was passed in. See parent for explanation.
	*
	* @param mixed
	*
	* @return array|false
	*/
	protected function _getExistingData($data)
	{
		if (!$id = $this->_getExistingPrimaryKey($data))
		{
			return false;
		}

		return array('xf_notification' => $this->_getNotificationModel()->getNotificationById($id));
	}

	/**
	* Gets SQL condition to update the existing record.
	*
	* @return string
	*/
	protected function _getUpdateCondition($tableName)
	{
		return 'notification_id = ' . $this->_db->quote($this->getExisting('notification_id'));
	}

	/**
	 * Verifies that the criteria is valid and formats is correctly.
	 * Expected input format: [] with children: [rule] => name, [data] => info
	 *
	 * @param array|string $criteria Criteria array or serialize string; see above for format. Modified by ref.
	 *
	 * @return boolean
	 */
	protected function _verifyCriteria(&$criteria)
	{
		$criteriaFiltered = XenForo_Helper_Criteria::prepareCriteriaForSave($criteria);
		$criteria = serialize($criteriaFiltered);
		return true;
	}

	/**
	 * Post-save handling.
	 */
	protected function _postSave()
	{
		$this->_rebuildNotificationCache();
	}

	/**
	 * Post-delete handling.
	 */
	protected function _postDelete()
	{
		$this->_db->delete('xf_notification_dismissed', 'notification_id = ' . $this->get('notification_id'));
		$this->_rebuildNotificationCache();
	}

	/**
	 * Rebuilds the notification cache.
	 */
	protected function _rebuildNotificationCache()
	{
		$this->_getNotificationModel()->rebuildNotificationCache();
	}

	/**
	 * @return XenForo_Model_Notification
	 */
	protected function _getNotificationModel()
	{
		return $this->getModelFromCache('GritterNotifications_Model_Notification');
	}
}